(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 8.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[    143061,       2950]
NotebookOptionsPosition[    142131,       2914]
NotebookOutlinePosition[    142476,       2929]
CellTagsIndexPosition[    142433,       2926]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell[TextData[{
 StyleBox["Practical Linear Algebra: A Geometry Toolbox, Third edition",
  FontWeight->"Bold",
  FontColor->RGBColor[0.7019607843137254, 0.34901960784313724`, 0.]],
 StyleBox["\nGerald Farin & Dianne Hansford\nCRC Press, Taylor & Francis \
Group, An A K Peters Book, 2013",
  FontColor->RGBColor[0.7019607843137254, 0.34901960784313724`, 0.]],
 "\n"
}], "Subtitle",
 CellChangeTimes->{{3.4991094606434994`*^9, 3.499109474090723*^9}, {
  3.5373646957107067`*^9, 3.53736471444634*^9}, {3.5374511355396433`*^9, 
  3.5374511374740467`*^9}, {3.5376327574376*^9, 3.5376327584984016`*^9}, {
  3.5842134956497355`*^9, 3.5842135862546945`*^9}},
 FontFamily->"Arial"],

Cell[TextData[{
 "\n",
 StyleBox["Projections in 2D",
  FontWeight->"Bold"],
 "\n\n",
 StyleBox["This notebook demonstrates projection matrices in 2D.\nA \
projection matrix P is rank deficient (determinant = 0) and it is idempotent \
(P = P^2), thus leaving the image unchanged on repeated applications of the \
map.\n\nWe construct a 2D projection matrix P by considering the map that \
takes the [e1,e2] vectors to the P = [a1, a2] vectors on a unit vector u, \
which defines the projection line.\n\n",
  FontFamily->"Arial"],
 StyleBox["Orthogonal projection:",
  FontFamily->"Arial",
  FontVariations->{"Underline"->True}],
 StyleBox["  \nSee Eqn (2.21) in PLA for details on construction.\na1 = \
(u.e1)u  = u_1u = [u_1^2  u_1u_2]^T \na2 = (u.e2).u = u_2u = [u_1u_2 u_2^2]^T\
\n\n",
  FontFamily->"Arial"],
 StyleBox["Oblique projection:",
  FontFamily->"Arial",
  FontVariations->{"Underline"->True}],
 StyleBox[" \nSee Section 5.3 in PLA for Cramer\[CloseCurlyQuote]s rule\nThe \
matrix will take the form P = [f1u, f2u].\nProject e1 and e2 onto u so an \
angle of alpha degrees is formed at the intersection with u. Solve for a1 and \
a2 via two linear systems. (v is a vector; c,d are scalars)\nLinear system \
for a1: e1 + cv = f1u\nVector v is created by a rotation of of u by alpha \
degrees.\nUse Cramer\[CloseCurlyQuote]s rule to solve for f1 = v_2/D, where D \
= sin(alpha) is the determinant of [v, -u]\nResult: a1 = v_2/D = f1u = [(u_1 \
sin(alpha) + u_2cos(alpha))/sin(alpha)]u\n\nSimilar idea for a2:\ne2 + cv = \
f2u (different v and c than above)\nForm v by a -alpha rotation of u.\na2 = \
-v_1/D = [(u_1 cos(alpha) + u_2sin(alpha))/sin(alpha)]u\n\nIdempotent and \
Determinant:\nDet[P] = 0\nP*P = (f1u_1 + f2u_2)P therefore, P is idempotent \
when (f1u_1 + f2u_2) = 1\nThis is the case when 2u_1u_2cos(alpha) = 0.\n\
Geometrically, this corresponds to oblique angle alpha = 90 or 270 degrees \
(an orthogonal projection) or a projection onto a coordinate axis such that \
u_1=0 or u_2=0.\nMatrices without the idempotent will flatten, but \
technically they are not projections.\n\n\nDetails of code below:\nThere are \
\[OpenCurlyDoubleQuote]numu\[CloseCurlyDoubleQuote] unit vectors (colored \
yellow) onto which e1 and e2 are projected. \nThe unit vectors e1 and e2 are \
black.\nThe vectors a1 and a2 are colored red and green, respectively.\nA \
blue line connects the end of e1 to a1 and e2 to a2.\nThe orthogonal code \
must be run before the oblique code. After that, the oblique code can be run \
with different angles (alpha).\nCode that prints out determinant and \
idempotent property is commented out.",
  FontFamily->"Arial"]
}], "Subsubtitle",
 CellChangeTimes->CompressedData["
1:eJxTTMoPSmViYGAQBmIQPbVOfGt45SvHOXsad4Fon8gdp0G0vqvQeRDtNcXr
Mog26Q+4CaK3WeQ+ANG9G+vA9JtnuU9B9CcTBoYIIL01yJANRC87/pUbRIc8
ucEPovftPW4IouXFFlmA6IRPe4pBdLLg70oQ7cCauhxE6/7gXQOiZ/zYuh5E
83AWXqsC0lGrQ6+DaLYV8vdAtF51BHs1kG5Qu8ANojeemZQAokU6f2WB6LIv
x4pA9Mf5J+pB9A3LqiYQXXP82q7L1a8c+dxugemOmrDvIFpGW/YHiBbmlN0t
V/PKMa9/2z4Q3Za9svvSFqB9d86BaQCgu55G
  "],
 FontSize->16],

Cell[CellGroupData[{

Cell["\<\

Orthogonal Projection\
\>", "Section",
 CellChangeTimes->{{3.553187568634225*^9, 3.553187583547851*^9}, {
  3.5532795368544264`*^9, 3.5532795454644403`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Select", " ", "number", " ", "of", " ", "unit", " ", "vectors", " ", "to",
     " ", "project", " ", "onto"}], " ", "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"numu", " ", "=", " ", "36"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"origin", " ", "=", " ", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0"}], "}"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"e1", " ", "=", " ", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"e2", " ", "=", " ", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Form", " ", "unit", " ", "vectors"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"degrees", " ", "=", " ", 
     RowBox[{"360", "/", "numu"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"u", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"RotationMatrix", "[", 
         RowBox[{"degrees", "*", 
          RowBox[{"(", 
           RowBox[{"i", "-", "1"}], ")"}], " ", "Degree"}], "]"}], ".", 
        "e1"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Form", " ", "columns", " ", "of", " ", "matrix", " ", "A"}], " ",
     "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"a1", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "1"}], "]"}], "]"}], "^", "2"}], ",", " ", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "1"}], "]"}], "]"}], "*", 
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "2"}], "]"}], "]"}]}]}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"a2", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "1"}], "]"}], "]"}], "*", 
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "2"}], "]"}], "]"}]}], ",", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "2"}], "]"}], "]"}], "^", "2"}]}], " ", "}"}], 
       ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Check", " ", "rank", " ", "and", " ", "idempotent", " ", "property", " ",
      "of", " ", "the", " ", "rth", " ", "matrix"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"Do", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{
        RowBox[{"mortho", " ", "=", " ", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"a1", "[", 
            RowBox[{"[", "i", "]"}], "]"}], ",", 
           RowBox[{"a2", "[", 
            RowBox[{"[", "i", "]"}], "]"}]}], "}"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Print", "[", 
          RowBox[{"\"\<Matrix orthogonal = \>\"", ",", 
           RowBox[{
            RowBox[{"MatrixForm", "[", "mortho", "]"}], "//", "N"}]}], "]"}], 
         "\[IndentingNewLine]", 
         RowBox[{"Print", "[", 
          RowBox[{"\"\<  Det = \>\"", ",", 
           RowBox[{"Det", "[", "mortho", "]"}], ",", "\"\<     P*P = \>\"", 
           ",", 
           RowBox[{
            RowBox[{"MatrixForm", "[", 
             RowBox[{"mortho", ".", "mortho"}], "]"}], "//", "N"}]}], 
          "]"}]}]}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}], ";"}], 
    "\[IndentingNewLine]", "*)"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"Print", "[", 
      RowBox[{"\"\<e1= \>\"", ",", "e1", ",", "\"\<   e2= \>\"", ",", "e2"}], 
      "]"}], "\[IndentingNewLine]", 
     RowBox[{"Print", "[", 
      RowBox[{
      "\"\<rotations 0 ... numu = \>\"", ",", "numu", ",", 
       "\"\<  degrees= \>\"", ",", "degrees"}], "]"}], "\[IndentingNewLine]", 
     RowBox[{"Print", "[", 
      RowBox[{"\"\<u = \>\"", ",", 
       RowBox[{"u", "//", "N"}]}], "]"}], "\[IndentingNewLine]", 
     RowBox[{"Print", "[", 
      RowBox[{"\"\<u[[numu]]= \>\"", ",", 
       RowBox[{"u", "[", 
        RowBox[{"[", "numu", "]"}], "]"}], ",", "\"\<  u[[numu,1]] = \>\"", 
       ",", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"[", 
          RowBox[{"numu", ",", "1"}], "]"}], "]"}], "//", "N"}], ",", 
       "\"\< u[[numu,2]] = \>\"", ",", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"[", 
          RowBox[{"numu", ",", "2"}], "]"}], "]"}], "//", "N"}]}], "]"}]}], 
    "\[IndentingNewLine]", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Print", "[", 
      RowBox[{"\"\<a1 = \>\"", ",", 
       RowBox[{"MatrixForm", "[", "a1", "]"}], ",", "\"\<  a2 = \>\"", ",", 
       RowBox[{"MatrixForm", "[", "a2", "]"}]}], "]"}], ";"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"SetDirectory", "[", 
     RowBox[{"NotebookDirectory", "[", "]"}], "]"}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Graphics", " ", "constructions"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"e1e2", " ", "=", " ", 
     RowBox[{"Graphics", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Thickness", "[", "0.006", "]"}], ",", "Black", ",", " ", 
        RowBox[{"Arrowheads", "[", "Large", "]"}], ",", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", "e1"}], "}"}], "]"}], ",", " ", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", "e2"}], "}"}], "]"}]}], "}"}], "]"}]}], 
    ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"uarrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.003", "]"}], ",", " ", "Yellow", ",", 
          RowBox[{"Arrowheads", "[", "Small", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"u", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"thickuarrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.006", "]"}], ",", " ", "Yellow", ",", 
          RowBox[{"Arrowheads", "[", "Small", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"u", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"a1arrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.007", "]"}], ",", " ", "Red", ",", 
          RowBox[{"Arrowheads", "[", "Medium", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"a1", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"a2arrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.007", "]"}], ",", " ", "Green", ",", 
          RowBox[{"Arrowheads", "[", "Medium", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"a2", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"e1e2line", " ", "=", " ", 
     RowBox[{"Graphics", "[", 
      RowBox[{"{", 
       RowBox[{"Thin", ",", " ", "Blue", ",", " ", 
        RowBox[{"Line", "[", 
         RowBox[{"{", 
          RowBox[{"e1", ",", "e2"}], "}"}], "]"}]}], "}"}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"Show", "[", 
    RowBox[{
    "e1e2", ",", " ", "uarrows", ",", " ", "a1arrows", ",", " ", "a2arrows"}],
     "]"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Display", " ", "one", " ", "mapping", " ", "and", " ", "connect", " ", 
     "e1", " ", "to", " ", "a1", " ", "and", " ", "e2", " ", "to", " ", 
     "a2"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"connect1", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{"Thin", ",", " ", "Red", ",", " ", 
          RowBox[{"Line", "[", 
           RowBox[{"{", 
            RowBox[{"e1", ",", 
             RowBox[{"a1", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"connect2", " ", "=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{"Thin", ",", " ", "Green", ",", " ", 
          RowBox[{"Line", "[", 
           RowBox[{"{", 
            RowBox[{"e2", ",", 
             RowBox[{"a2", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"Manipulate", "[", 
    RowBox[{
     RowBox[{"Show", "[", 
      RowBox[{"e1e2", ",", " ", "uarrows", ",", " ", 
       RowBox[{"thickuarrows", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}], ",", 
       RowBox[{"a1arrows", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}], ",", 
       RowBox[{"a2arrows", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}], ",", " ", 
       RowBox[{"connect1", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}], ",", " ", 
       RowBox[{"connect2", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}]}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"uvec", ",", " ", 
         RowBox[{"IntegerPart", "[", 
          RowBox[{"numu", "/", "6"}], "]"}]}], "}"}], ",", " ", "1", ",", 
       "numu", ",", "1"}], "}"}]}], "]"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.5531702152149353`*^9, 3.553170225401753*^9}, {
   3.553170256336607*^9, 3.5531702895178657`*^9}, {3.553170329032735*^9, 
   3.553170345911965*^9}, {3.5531727803274403`*^9, 3.5531727938214645`*^9}, {
   3.5531729213740883`*^9, 3.553173028717877*^9}, 3.553173080665968*^9, {
   3.5531731352660637`*^9, 3.5531731768245373`*^9}, {3.553173333402012*^9, 
   3.5531733568956532`*^9}, {3.5531734099669466`*^9, 3.553173488622285*^9}, {
   3.553173573205633*^9, 3.5531736244829235`*^9}, {3.5531736569153805`*^9, 
   3.5531736592553844`*^9}, {3.553173826596878*^9, 3.5531738733969603`*^9}, {
   3.5531739538931017`*^9, 3.55317397557714*^9}, {3.5531740090859985`*^9, 
   3.5531740554648805`*^9}, {3.5531740892389393`*^9, 3.553174090627342*^9}, {
   3.553174128129808*^9, 3.553174259185638*^9}, {3.5531744312695403`*^9, 
   3.553174519721696*^9}, {3.5531745533865547`*^9, 3.5531745628713713`*^9}, {
   3.5531746028542414`*^9, 3.553174694644803*^9}, {3.5531747767477474`*^9, 
   3.5531748042973957`*^9}, {3.5531749027491684`*^9, 
   3.5531749173975945`*^9}, {3.553175072025066*^9, 3.5531751445027933`*^9}, {
   3.5531751816620584`*^9, 3.5531752040792975`*^9}, {3.553175246558172*^9, 
   3.553175251596981*^9}, {3.553175284310239*^9, 3.553175316539895*^9}, {
   3.5531755237550592`*^9, 3.5531755847979665`*^9}, {3.55317562103683*^9, 
   3.553175643079669*^9}, {3.5531757083345833`*^9, 3.553175965735036*^9}, {
   3.5531760499595833`*^9, 3.55317622452389*^9}, {3.553176303491229*^9, 
   3.553176346094904*^9}, {3.553176424719042*^9, 3.5531764673227167`*^9}, {
   3.553176500191974*^9, 3.5531765185844064`*^9}, {3.553176553091667*^9, 
   3.553176569705696*^9}, {3.553176619641384*^9, 3.553176677533086*^9}, {
   3.5531767098719425`*^9, 3.553176710573944*^9}, 3.5531773284130287`*^9, {
   3.553187556825004*^9, 3.5531875615362124`*^9}, {3.5531882012921357`*^9, 
   3.553188202680538*^9}, {3.553188517847892*^9, 3.5531885179570923`*^9}, 
   3.553189499386016*^9, {3.553195014822503*^9, 3.553195095490245*^9}, {
   3.553195176547987*^9, 3.5531951780299892`*^9}, {3.5531978282487154`*^9, 
   3.553197872287593*^9}, {3.553197943330117*^9, 3.5531979436265182`*^9}, {
   3.553198009209033*^9, 3.5531980569763174`*^9}, 3.553198105336402*^9, {
   3.5531981386892605`*^9, 3.553198148252077*^9}, {3.5531981785317307`*^9, 
   3.5531981913237534`*^9}, {3.5531982320242243`*^9, 3.553198249355855*^9}, {
   3.553198284861517*^9, 3.5531983777596807`*^9}, {3.5531984103169374`*^9, 
   3.5531984570078197`*^9}, {3.5531985314355507`*^9, 3.553198610231289*^9}, {
   3.5531991379022155`*^9, 3.553199138042616*^9}, {3.553199280018465*^9, 
   3.5531993099393177`*^9}, {3.5532732573374214`*^9, 3.5532732582266226`*^9}, 
   3.553276671305341*^9, 3.553276945697798*^9, {3.5532770293602414`*^9, 
   3.5532770299218426`*^9}, {3.553277306517887*^9, 3.553277307568689*^9}, {
   3.553281448383069*^9, 3.553281544083233*^9}, {3.5532818849094076`*^9, 
   3.5532818866594105`*^9}, {3.5532819388890967`*^9, 
   3.5532820207092366`*^9}, {3.5532820555768967`*^9, 
   3.5532823971844597`*^9}, {3.553282436877329*^9, 3.553282441261736*^9}, {
   3.553287356552853*^9, 3.553287359953659*^9}, {3.5532977839858212`*^9, 
   3.55329779482784*^9}, {3.5533526045821524`*^9, 3.5533526070937567`*^9}, {
   3.55335275712562*^9, 3.5533527572504196`*^9}, {3.559134063131215*^9, 
   3.559134064847218*^9}, {3.5591341546253757`*^9, 3.5591341567313795`*^9}, {
   3.559134326740478*^9, 3.5591343483465157`*^9}, {3.559134440043477*^9, 
   3.559134517482013*^9}, {3.5591345739853125`*^9, 3.5591345759821157`*^9}, {
   3.5591346095241685`*^9, 3.5591346473666244`*^9}, {3.5591346796738715`*^9, 
   3.559134707747913*^9}, {3.5591347573647842`*^9, 3.5591347688048*^9}, {
   3.559134809032858*^9, 3.5591348407941055`*^9}, {3.559134930614632*^9, 
   3.559135015454351*^9}, {3.5591350511744013`*^9, 3.559135061394415*^9}, {
   3.5591352663867073`*^9, 3.559135324442398*^9}, {3.559135357296056*^9, 
   3.5591353775760913`*^9}, {3.5591354242513733`*^9, 
   3.5591354271061783`*^9}, {3.5591354600066357`*^9, 
   3.5591354793038697`*^9}, {3.5591355764920406`*^9, 3.55913562188812*^9}, {
   3.5591356738518114`*^9, 3.559135680123022*^9}, {3.55913571824949*^9, 
   3.559135732476714*^9}, {3.5591357666095743`*^9, 3.559135904607417*^9}, {
   3.5591359424374833`*^9, 3.5591359719059353`*^9}, {3.5591360330412426`*^9, 
   3.5591360676109033`*^9}, {3.559136112367382*^9, 3.559136147639044*^9}, {
   3.559136225935581*^9, 3.559136269693658*^9}, {3.559136342436586*^9, 
   3.559136346788994*^9}, {3.5591363997510867`*^9, 3.559136399875887*^9}, {
   3.5591364329479446`*^9, 3.5591365017596655`*^9}, {3.5591365385445304`*^9, 
   3.55913660656065*^9}, {3.5591366445779166`*^9, 3.559136678929177*^9}, {
   3.559136727819663*^9, 3.5591367900457726`*^9}, {3.559136824615433*^9, 
   3.5591368277666388`*^9}, {3.5591371328251743`*^9, 
   3.5591371332775755`*^9}, {3.5591385625578904`*^9, 3.559138563946293*^9}, {
   3.559138702271736*^9, 3.5591387349381933`*^9}, {3.559138952340175*^9, 
   3.559138973743413*^9}, {3.5591390079698725`*^9, 3.5591390112770786`*^9}, {
   3.5591516368200607`*^9, 3.5591516369292603`*^9}, {3.5629353049953346`*^9, 
   3.562935323387767*^9}, {3.5701354249303617`*^9, 3.570135425991164*^9}}],

Cell[BoxData[
 GraphicsBox[{
   {GrayLevel[0], Thickness[0.006], Arrowheads[Large], 
    ArrowBox[{{0, 0}, {1, 0}}], ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[{{0, 0}, {1, 0}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[10 Degree], Sin[10 Degree]}}, {{0, 0}, {
       0.984807753012208, 0.17364817766693033`}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[20 Degree], Sin[20 Degree]}}, {{0, 0}, {
       0.9396926207859084, 0.3420201433256687}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 2] 3^Rational[1, 2], Rational[1, 2]}}, {{0, 0}, {
       0.8660254037844386, 0.5}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[40 Degree], Sin[40 Degree]}}, {{0, 0}, {
       0.766044443118978, 0.6427876096865393}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[40 Degree], Cos[40 Degree]}}, {{0, 0}, {
       0.6427876096865393, 0.766044443118978}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {{0, 0}, {0.5, 
       0.8660254037844386}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[20 Degree], Cos[20 Degree]}}, {{0, 0}, {
       0.3420201433256687, 0.9396926207859084}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[10 Degree], Cos[10 Degree]}}, {{0, 0}, {
       0.17364817766693033`, 0.984807753012208}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[10 Degree], Cos[10 Degree]}}, {{0, 
       0}, {-0.17364817766693033`, 0.984807753012208}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[20 Degree], Cos[20 Degree]}}, {{0, 
       0}, {-0.3420201433256687, 0.9396926207859084}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {{0, 0}, {-0.5, 
       0.8660254037844386}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[40 Degree], Cos[40 Degree]}}, {{0, 
       0}, {-0.6427876096865393, 0.766044443118978}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[40 Degree], Sin[40 Degree]}}, {{0, 
       0}, {-0.766044443118978, 0.6427876096865393}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 2] 3^Rational[1, 2], Rational[1, 2]}}, {{0, 
       0}, {-0.8660254037844386, 0.5}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[20 Degree], Sin[20 Degree]}}, {{0, 
       0}, {-0.9396926207859084, 0.3420201433256687}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[10 Degree], Sin[10 Degree]}}, {{0, 
       0}, {-0.984807753012208, 0.17364817766693033`}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[{{0, 0}, {-1, 0}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[10 Degree], -Sin[10 Degree]}}, {{0, 
       0}, {-0.984807753012208, -0.17364817766693033`}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[20 Degree], -Sin[20 Degree]}}, {{0, 
       0}, {-0.9396926207859084, -0.3420201433256687}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 2] 3^Rational[1, 2], Rational[-1, 2]}}, {{0, 
       0}, {-0.8660254037844386, -0.5}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[40 Degree], -Sin[40 Degree]}}, {{0, 
       0}, {-0.766044443118978, -0.6427876096865393}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[40 Degree], -Cos[40 Degree]}}, {{0, 
       0}, {-0.6427876096865393, -0.766044443118978}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 2], Rational[-1, 2] 3^Rational[1, 2]}}, {{0, 
       0}, {-0.5, -0.8660254037844386}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[20 Degree], -Cos[20 Degree]}}, {{0, 
       0}, {-0.3420201433256687, -0.9396926207859084}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[10 Degree], -Cos[10 Degree]}}, {{0, 
       0}, {-0.17364817766693033`, -0.984807753012208}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[{{0, 0}, {0, -1}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[10 Degree], -Cos[10 Degree]}}, {{0, 0}, {
       0.17364817766693033`, -0.984807753012208}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[20 Degree], -Cos[20 Degree]}}, {{0, 0}, {
       0.3420201433256687, -0.9396926207859084}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 2], Rational[-1, 2] 3^Rational[1, 2]}}, {{0, 0}, {
       0.5, -0.8660254037844386}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[40 Degree], -Cos[40 Degree]}}, {{0, 0}, {
       0.6427876096865393, -0.766044443118978}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[
     NCache[{{0, 0}, {Cos[40 Degree], -Sin[40 Degree]}}, {{0, 0}, {
       0.766044443118978, -0.6427876096865393}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 2] 3^Rational[1, 2], Rational[-1, 2]}}, {{0, 0}, {
       0.8660254037844386, -0.5}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[20 Degree], -Sin[20 Degree]}}, {{0, 0}, {
       0.9396926207859084, -0.3420201433256687}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[10 Degree], -Sin[10 Degree]}}, {{0, 0}, {
       0.984807753012208, -0.17364817766693033`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {1, 0}}]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[10 Degree]^2, Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
       0.9698463103929541, 0.17101007166283433`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[20 Degree]^2, Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
       0.8830222215594891, 0.3213938048432697}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[3, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {0.75, 
       0.4330127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[40 Degree]^2, Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
       0.5868240888334652, 0.49240387650610395`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[40 Degree]^2, Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
       0.41317591116653474`, 0.49240387650610395`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {0.25, 
       0.4330127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[20 Degree]^2, Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
       0.11697777844051097`, 0.3213938048432697}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[10 Degree]^2, Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
       0.030153689607045803`, 0.17101007166283433`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 0}}]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[10 Degree]^2, -Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
       0.030153689607045803`, -0.17101007166283433`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[20 Degree]^2, -Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
       0.11697777844051097`, -0.3213938048432697}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 4], Rational[-1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
       0.25, -0.4330127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[40 Degree]^2, -Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
       0.41317591116653474`, -0.49240387650610395`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[40 Degree]^2, -Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
       0.5868240888334652, -0.49240387650610395`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[3, 4], Rational[-1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
       0.75, -0.4330127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[20 Degree]^2, -Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
       0.8830222215594891, -0.3213938048432697}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[10 Degree]^2, -Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
       0.9698463103929541, -0.17101007166283433`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {1, 0}}]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[10 Degree]^2, Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
       0.9698463103929541, 0.17101007166283433`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[20 Degree]^2, Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
       0.8830222215594891, 0.3213938048432697}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[3, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {0.75, 
       0.4330127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[40 Degree]^2, Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
       0.5868240888334652, 0.49240387650610395`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[40 Degree]^2, Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
       0.41317591116653474`, 0.49240387650610395`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {0.25, 
       0.4330127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[20 Degree]^2, Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
       0.11697777844051097`, 0.3213938048432697}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[10 Degree]^2, Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
       0.030153689607045803`, 0.17101007166283433`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 0}}]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[10 Degree]^2, -Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
       0.030153689607045803`, -0.17101007166283433`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[20 Degree]^2, -Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
       0.11697777844051097`, -0.3213938048432697}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 4], Rational[-1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
       0.25, -0.4330127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Sin[40 Degree]^2, -Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
       0.41317591116653474`, -0.49240387650610395`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[40 Degree]^2, -Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
       0.5868240888334652, -0.49240387650610395`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[3, 4], Rational[-1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
       0.75, -0.4330127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[20 Degree]^2, -Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
       0.8830222215594891, -0.3213938048432697}}]]}, 
   {RGBColor[1, 0, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[10 Degree]^2, -Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
       0.9698463103929541, -0.17101007166283433`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 0}}]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[10 Degree] Sin[10 Degree], Sin[10 Degree]^2}}, {{0, 0}, {
       0.17101007166283433`, 0.030153689607045803`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[20 Degree] Sin[20 Degree], Sin[20 Degree]^2}}, {{0, 0}, {
       0.3213938048432697, 0.11697777844051097`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 0}, {
       0.4330127018922193, 0.25}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[40 Degree] Sin[40 Degree], Sin[40 Degree]^2}}, {{0, 0}, {
       0.49240387650610395`, 0.41317591116653474`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[40 Degree] Sin[40 Degree], Cos[40 Degree]^2}}, {{0, 0}, {
       0.49240387650610395`, 0.5868240888334652}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 4] 3^Rational[1, 2], Rational[3, 4]}}, {{0, 0}, {
       0.4330127018922193, 0.75}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[20 Degree] Sin[20 Degree], Cos[20 Degree]^2}}, {{0, 0}, {
       0.3213938048432697, 0.8830222215594891}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[10 Degree] Sin[10 Degree], Cos[10 Degree]^2}}, {{0, 0}, {
       0.17101007166283433`, 0.9698463103929541}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[10 Degree] Sin[10 Degree], Cos[10 Degree]^2}}, {{0, 
       0}, {-0.17101007166283433`, 0.9698463103929541}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[20 Degree] Sin[20 Degree], Cos[20 Degree]^2}}, {{0, 
       0}, {-0.3213938048432697, 0.8830222215594891}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[
     NCache[{{0, 0}, {Rational[-1, 4] 3^Rational[1, 2], Rational[3, 4]}}, {{0,
        0}, {-0.4330127018922193, 0.75}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[40 Degree] Sin[40 Degree], Cos[40 Degree]^2}}, {{0, 
       0}, {-0.49240387650610395`, 0.5868240888334652}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[40 Degree] Sin[40 Degree], Sin[40 Degree]^2}}, {{0, 
       0}, {-0.49240387650610395`, 0.41317591116653474`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 
       0}, {-0.4330127018922193, 0.25}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[20 Degree] Sin[20 Degree], Sin[20 Degree]^2}}, {{0, 
       0}, {-0.3213938048432697, 0.11697777844051097`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[10 Degree] Sin[10 Degree], Sin[10 Degree]^2}}, {{0, 
       0}, {-0.17101007166283433`, 0.030153689607045803`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 0}}]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[10 Degree] Sin[10 Degree], Sin[10 Degree]^2}}, {{0, 0}, {
       0.17101007166283433`, 0.030153689607045803`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[20 Degree] Sin[20 Degree], Sin[20 Degree]^2}}, {{0, 0}, {
       0.3213938048432697, 0.11697777844051097`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 0}, {
       0.4330127018922193, 0.25}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[40 Degree] Sin[40 Degree], Sin[40 Degree]^2}}, {{0, 0}, {
       0.49240387650610395`, 0.41317591116653474`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[40 Degree] Sin[40 Degree], Cos[40 Degree]^2}}, {{0, 0}, {
       0.49240387650610395`, 0.5868240888334652}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 4] 3^Rational[1, 2], Rational[3, 4]}}, {{0, 0}, {
       0.4330127018922193, 0.75}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[20 Degree] Sin[20 Degree], Cos[20 Degree]^2}}, {{0, 0}, {
       0.3213938048432697, 0.8830222215594891}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Cos[10 Degree] Sin[10 Degree], Cos[10 Degree]^2}}, {{0, 0}, {
       0.17101007166283433`, 0.9698463103929541}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[10 Degree] Sin[10 Degree], Cos[10 Degree]^2}}, {{0, 
       0}, {-0.17101007166283433`, 0.9698463103929541}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[20 Degree] Sin[20 Degree], Cos[20 Degree]^2}}, {{0, 
       0}, {-0.3213938048432697, 0.8830222215594891}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 4] 3^Rational[1, 2], Rational[3, 4]}}, {{0, 
       0}, {-0.4330127018922193, 0.75}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[40 Degree] Sin[40 Degree], Cos[40 Degree]^2}}, {{0, 
       0}, {-0.49240387650610395`, 0.5868240888334652}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[40 Degree] Sin[40 Degree], Sin[40 Degree]^2}}, {{0, 
       0}, {-0.49240387650610395`, 0.41317591116653474`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 
       0}, {-0.4330127018922193, 0.25}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[20 Degree] Sin[20 Degree], Sin[20 Degree]^2}}, {{0, 
       0}, {-0.3213938048432697, 0.11697777844051097`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[0.007], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Cos[10 Degree] Sin[10 Degree], Sin[10 Degree]^2}}, {{0, 
       0}, {-0.17101007166283433`, 0.030153689607045803`}}]]}}]], "Output",
 CellChangeTimes->{
  3.5531882035697393`*^9, 3.5531885189242935`*^9, 3.553189500774418*^9, 
   3.553195124178695*^9, 3.5531951789503913`*^9, 3.553197893004429*^9, 
   3.5531979448277197`*^9, {3.553198045011096*^9, 3.553198058177519*^9}, {
   3.553198099267991*^9, 3.5531981076296062`*^9}, 3.5531981494376793`*^9, 
   3.553198194116158*^9, 3.553198250447857*^9, 3.553198289853526*^9, {
   3.553198437960186*^9, 3.553198459051423*^9}, 3.553198555833993*^9, {
   3.55319859948287*^9, 3.55319861096449*^9}, 3.5531991421922235`*^9, 
   3.553199310828519*^9, 3.5531997420444765`*^9, 3.5532608130649724`*^9, 
   3.5532610325351562`*^9, 3.5532612693665504`*^9, 3.5532729847217646`*^9, 
   3.5532766778885527`*^9, 3.5532769501938057`*^9, 3.553277034104649*^9, 
   3.553277310896294*^9, 3.553281571377678*^9, 3.5532819701523514`*^9, {
   3.55328201109962*^9, 3.553282023049241*^9}, 3.553282092962364*^9, 
   3.5532821500480585`*^9, 3.5532822044181347`*^9, {3.5532822559478197`*^9, 
   3.5532822892734766`*^9}, 3.5532823989536624`*^9, 3.5532824700201855`*^9, 
   3.553287361201661*^9, 3.5532875725548325`*^9, 3.553287777164792*^9, 
   3.5532975450558014`*^9, 3.5532977994142485`*^9, 3.553352565344884*^9, {
   3.553352741026391*^9, 3.5533527609944267`*^9}, 3.554581742177843*^9, 
   3.5545823320148787`*^9, 3.554582364650136*^9, 3.554582399859398*^9, 
   3.5547357488040595`*^9, 3.5591339134489517`*^9, 3.559134066563221*^9, 
   3.559134649355027*^9, 3.559134776654811*^9, 3.559134841385306*^9, 
   3.559135016774353*^9, {3.5591352488966827`*^9, 3.5591352676167088`*^9}, 
   3.5591353256904*^9, 3.559135378808493*^9, 3.5591354285881805`*^9, {
   3.559135466277847*^9, 3.5591354810042725`*^9}, {3.559135586335658*^9, 
   3.559135622948922*^9}, 3.5591356807470236`*^9, {3.5591357194662914`*^9, 
   3.559135733287916*^9}, 3.5591357677327766`*^9, 3.5591359060270195`*^9, 
   3.559135972935537*^9, 3.5591361585434628`*^9, 3.559136281331279*^9, 
   3.559136348270996*^9, 3.5591364015138893`*^9, 3.5591364611839943`*^9, 
   3.559136503631669*^9, {3.559136582115407*^9, 3.559136607855452*^9}, 
   3.559136681050781*^9, {3.5591367644929276`*^9, 3.5591367910597744`*^9}, 
   3.559136829919442*^9, 3.559137134353977*^9, 3.559138740991004*^9, 
   3.5591388026891127`*^9, {3.5591389559749813`*^9, 3.5591389753034153`*^9}, 
   3.559139014459484*^9, 3.559139136373698*^9, 3.559139295702779*^9, 
   3.559151643278472*^9, 3.559151682824541*^9, 3.5591517501698594`*^9, {
   3.5591517802935123`*^9, 3.559151803038352*^9}, 3.5591518909133067`*^9, 
   3.570135134592252*^9, 3.570135431794374*^9, 3.570136956447452*^9, 
   3.5844640075548496`*^9}],

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`uvec$$ = 7, Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`uvec$$], 6}, 1, 36, 1}}, Typeset`size$$ = {
    360., {178., 181.}}, Typeset`update$$ = 0, Typeset`initDone$$, 
    Typeset`skipInitDone$$ = True, $CellContext`uvec$1634$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, "Variables" :> {$CellContext`uvec$$ = 6}, 
      "ControllerVariables" :> {
        Hold[$CellContext`uvec$$, $CellContext`uvec$1634$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> 
      Show[$CellContext`e1e2, $CellContext`uarrows, 
        Part[$CellContext`thickuarrows, $CellContext`uvec$$], 
        Part[$CellContext`a1arrows, $CellContext`uvec$$], 
        Part[$CellContext`a2arrows, $CellContext`uvec$$], 
        Part[$CellContext`connect1, $CellContext`uvec$$], 
        Part[$CellContext`connect2, $CellContext`uvec$$]], 
      "Specifications" :> {{{$CellContext`uvec$$, 6}, 1, 36, 1}}, 
      "Options" :> {}, "DefaultOptions" :> {}],
     ImageSizeCache->{409., {226., 231.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellChangeTimes->{
  3.5531882035697393`*^9, 3.5531885189242935`*^9, 3.553189500774418*^9, 
   3.553195124178695*^9, 3.5531951789503913`*^9, 3.553197893004429*^9, 
   3.5531979448277197`*^9, {3.553198045011096*^9, 3.553198058177519*^9}, {
   3.553198099267991*^9, 3.5531981076296062`*^9}, 3.5531981494376793`*^9, 
   3.553198194116158*^9, 3.553198250447857*^9, 3.553198289853526*^9, {
   3.553198437960186*^9, 3.553198459051423*^9}, 3.553198555833993*^9, {
   3.55319859948287*^9, 3.55319861096449*^9}, 3.5531991421922235`*^9, 
   3.553199310828519*^9, 3.5531997420444765`*^9, 3.5532608130649724`*^9, 
   3.5532610325351562`*^9, 3.5532612693665504`*^9, 3.5532729847217646`*^9, 
   3.5532766778885527`*^9, 3.5532769501938057`*^9, 3.553277034104649*^9, 
   3.553277310896294*^9, 3.553281571377678*^9, 3.5532819701523514`*^9, {
   3.55328201109962*^9, 3.553282023049241*^9}, 3.553282092962364*^9, 
   3.5532821500480585`*^9, 3.5532822044181347`*^9, {3.5532822559478197`*^9, 
   3.5532822892734766`*^9}, 3.5532823989536624`*^9, 3.5532824700201855`*^9, 
   3.553287361201661*^9, 3.5532875725548325`*^9, 3.553287777164792*^9, 
   3.5532975450558014`*^9, 3.5532977994142485`*^9, 3.553352565344884*^9, {
   3.553352741026391*^9, 3.5533527609944267`*^9}, 3.554581742177843*^9, 
   3.5545823320148787`*^9, 3.554582364650136*^9, 3.554582399859398*^9, 
   3.5547357488040595`*^9, 3.5591339134489517`*^9, 3.559134066563221*^9, 
   3.559134649355027*^9, 3.559134776654811*^9, 3.559134841385306*^9, 
   3.559135016774353*^9, {3.5591352488966827`*^9, 3.5591352676167088`*^9}, 
   3.5591353256904*^9, 3.559135378808493*^9, 3.5591354285881805`*^9, {
   3.559135466277847*^9, 3.5591354810042725`*^9}, {3.559135586335658*^9, 
   3.559135622948922*^9}, 3.5591356807470236`*^9, {3.5591357194662914`*^9, 
   3.559135733287916*^9}, 3.5591357677327766`*^9, 3.5591359060270195`*^9, 
   3.559135972935537*^9, 3.5591361585434628`*^9, 3.559136281331279*^9, 
   3.559136348270996*^9, 3.5591364015138893`*^9, 3.5591364611839943`*^9, 
   3.559136503631669*^9, {3.559136582115407*^9, 3.559136607855452*^9}, 
   3.559136681050781*^9, {3.5591367644929276`*^9, 3.5591367910597744`*^9}, 
   3.559136829919442*^9, 3.559137134353977*^9, 3.559138740991004*^9, 
   3.5591388026891127`*^9, {3.5591389559749813`*^9, 3.5591389753034153`*^9}, 
   3.559139014459484*^9, 3.559139136373698*^9, 3.559139295702779*^9, 
   3.559151643278472*^9, 3.559151682824541*^9, 3.5591517501698594`*^9, {
   3.5591517802935123`*^9, 3.559151803038352*^9}, 3.5591518909133067`*^9, 
   3.570135134592252*^9, 3.570135431794374*^9, 3.570136956447452*^9, 
   3.58446400766405*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{"Figure", " ", "for", " ", "book"}], " ", "*)"}], 
  "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Redefine", " ", "stuff", " ", "from", " ", "above", " ", "with", " ", 
    "gray", " ", "scales", " ", "only"}], " ", "*)"}], "\[IndentingNewLine]", 
  "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
    RowBox[{"use", " ", "numu"}], "=", 
    RowBox[{
     RowBox[{"36", " ", "and", " ", "iarrow"}], "=", 
     RowBox[{
     "6", " ", "to", " ", "get", " ", "45", " ", "degree", " ", "example", 
      " ", "that", " ", "is", " ", "in", " ", "Projections", " ", "2", 
      "d"}]}]}], " ", "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Figure", " ", "with", " ", "just", " ", "one", " ", "projection"}], " ", 
   "*)"}], "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "drawing", " ", "a2", " ", "after", " ", "a1", " ", "so", " ", "appears", 
    " ", "on", " ", "top", " ", "in", " ", "this", " ", "example"}], " ", 
   "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"iarrow", " ", "=", " ", "4"}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"plotarrows", " ", "=", " ", 
     RowBox[{"Graphics", "[", 
      RowBox[{"{", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Thickness", "[", "0.006", "]"}], ",", " ", 
        RowBox[{"GrayLevel", "[", "0.9", "]"}], ",", " ", 
        RowBox[{"Arrowheads", "[", "Medium", "]"}], ",", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", "e1"}], "}"}], "]"}], ",", " ", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", "e2"}], "}"}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"Thickness", "[", "0.006", "]"}], ",", " ", 
        RowBox[{"GrayLevel", "[", "0.7", "]"}], ",", 
        RowBox[{"Arrowheads", "[", "Large", "]"}], ",", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", 
           RowBox[{"u", "[", 
            RowBox[{"[", "iarrow", "]"}], "]"}]}], "}"}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"GrayLevel", "[", "0.0", "]"}], ",", "\[IndentingNewLine]", 
        RowBox[{"Text", "[", 
         RowBox[{
          RowBox[{"Style", "[", 
           RowBox[{"\"\<u\>\"", ",", 
            RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}], ",", 
            "Bold", ",", "18"}], "]"}], ",", 
          RowBox[{"u", "[", 
           RowBox[{"[", "iarrow", "]"}], "]"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", 
            RowBox[{"-", "1.0"}]}], "}"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"Thickness", "[", "0.008", "]"}], ",", " ", 
        RowBox[{"GrayLevel", "[", "0.0", "]"}], ",", 
        RowBox[{"Arrowheads", "[", "Large", "]"}], ",", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", 
           RowBox[{"a1", "[", 
            RowBox[{"[", "iarrow", "]"}], "]"}]}], "}"}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"Thickness", "[", "0.008", "]"}], ",", " ", 
        RowBox[{"GrayLevel", "[", "0.5", "]"}], ",", 
        RowBox[{"Arrowheads", "[", "Large", "]"}], ",", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", 
           RowBox[{"a2", "[", 
            RowBox[{"[", "iarrow", "]"}], "]"}]}], "}"}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"Thickness", "[", "0.002", "]"}], ",", " ", 
        RowBox[{"GrayLevel", "[", "0.0", "]"}], ",", " ", 
        RowBox[{"Line", "[", 
         RowBox[{"{", 
          RowBox[{"e1", ",", 
           RowBox[{"a1", "[", 
            RowBox[{"[", "iarrow", "]"}], "]"}]}], "}"}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"Thickness", "[", "0.002", "]"}], ",", " ", 
        RowBox[{"GrayLevel", "[", "0.5", "]"}], ",", " ", 
        RowBox[{"Line", "[", 
         RowBox[{"{", 
          RowBox[{"e2", ",", 
           RowBox[{"a2", "[", 
            RowBox[{"[", "iarrow", "]"}], "]"}]}], "}"}], "]"}]}], 
       "\[IndentingNewLine]", "}"}], "]"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Figure", " ", "with", " ", "all", " ", "the", " ", "projections"}], "  ",
     "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"be1e2", " ", "=", " ", 
     RowBox[{"Graphics", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Thickness", "[", "0.006", "]"}], ",", " ", 
        RowBox[{"GrayLevel", "[", "0.9", "]"}], ",", " ", 
        RowBox[{"Arrowheads", "[", "Medium", "]"}], ",", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", "e1"}], "}"}], "]"}], ",", " ", 
        RowBox[{"Arrow", "[", 
         RowBox[{"{", 
          RowBox[{"origin", ",", " ", "e2"}], "}"}], "]"}]}], "}"}], "]"}]}], 
    ";"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"was", " ", "Yellow"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"buarrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.003", "]"}], ",", " ", 
          RowBox[{"GrayLevel", "[", "0.7", "]"}], ",", 
          RowBox[{"Arrowheads", "[", "Small", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"u", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"was", " ", "Red"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"ba1arrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.007", "]"}], ",", " ", 
          RowBox[{"GrayLevel", "[", "0.0", "]"}], ",", 
          RowBox[{"Arrowheads", "[", "Medium", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"a1", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"was", " ", "Green"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"ba2arrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.007", "]"}], ",", " ", 
          RowBox[{"GrayLevel", "[", "0.5", "]"}], ",", 
          RowBox[{"Arrowheads", "[", "Medium", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"a2", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"picall", " ", "=", " ", 
     RowBox[{"Show", "[", 
      RowBox[{
      "be1e2", ",", " ", "buarrows", ",", " ", "ba1arrows", ",", " ", 
       "ba2arrows"}], "]"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"picbook", " ", "=", " ", 
    RowBox[{"GraphicsRow", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"plotarrows", ",", " ", "picall"}], "}"}], ",", 
      RowBox[{"Alignment", "\[Rule]", "Center"}], ",", " ", 
      RowBox[{"ImageSize", "\[Rule]", "Full"}]}], "]"}]}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Export", "[", 
     RowBox[{"\"\<Projections2D.eps\>\"", ",", " ", "picbook"}], "]"}], 
    ";"}]}]}]], "Input",
 CellChangeTimes->{{3.55913854533546*^9, 3.5591385506394696`*^9}, {
  3.559138599436355*^9, 3.5591386877637105`*^9}, {3.5591387970419025`*^9, 
  3.5591387995379066`*^9}, {3.559138860565214*^9, 3.559138911733304*^9}, {
  3.559139055659157*^9, 3.5591390774679947`*^9}, {3.559139267451129*^9, 
  3.559139284595559*^9}, {3.559151678144533*^9, 3.559151678565734*^9}, {
  3.559151746472653*^9, 3.559151799559546*^9}, {3.559151886436099*^9, 
  3.5591518869040995`*^9}, {3.5701354138231425`*^9, 3.5701354144627438`*^9}, {
  3.5701358926035833`*^9, 3.570135928592847*^9}, {3.5701359898541546`*^9, 
  3.5701360766059065`*^9}, {3.570136116151976*^9, 3.5701361311124024`*^9}, {
  3.5701363105439177`*^9, 3.570136367983218*^9}, {3.5701367598715067`*^9, 
  3.5701367618995104`*^9}, {3.5701368310388317`*^9, 3.570136856108076*^9}, {
  3.570136901628956*^9, 3.5701369159653807`*^9}, {3.570136950706642*^9, 
  3.5701369509718423`*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {InsetBox[
     GraphicsBox[{
       {GrayLevel[0.9], Thickness[0.006], Arrowheads[Medium], 
        ArrowBox[{{0, 0}, {1, 0}}], ArrowBox[{{0, 0}, {0, 1}}]}, 
       {GrayLevel[0.7], Thickness[0.006], Arrowheads[Large], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 2] 3^Rational[1, 2], Rational[1, 2]}}, {{0, 0}, {
           0.8660254037844386, 0.5}}]], 
        {GrayLevel[0.], 
         StyleBox[InsetBox[
           StyleBox["\<\"u\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->18,
            FontWeight->Bold], 
           NCache[{Rational[1, 2] 3^Rational[1, 2], Rational[1, 2]}, {
            0.8660254037844386, 0.5}], {0, -1.}],
          FontColor->GrayLevel[0.]]}, 
        {GrayLevel[0.], Thickness[0.008], Arrowheads[Large], 
         StyleBox[
          ArrowBox[
           NCache[{{0, 0}, {
             Rational[3, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
             0.75, 0.4330127018922193}}]],
          FontColor->GrayLevel[0.]]}, 
        {GrayLevel[0.5], Thickness[0.008], Arrowheads[Large], 
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 0}, {
            0.4330127018922193, 0.25}}]], 
         {GrayLevel[0.], Thickness[0.002], 
          
          StyleBox[
           LineBox[NCache[{{1, 0}, {
              Rational[3, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{1, 0}, {
              0.75, 0.4330127018922193}}]],
           FontColor->GrayLevel[0.]]}, 
         {GrayLevel[0.5], Thickness[0.002], 
          LineBox[NCache[{{0, 1}, {
             Rational[1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 1}, {
             0.4330127018922193, 0.25}}]]}}}}], {192., -189.}, 
     ImageScaled[{0.5, 0.5}], {360., 360.}], InsetBox[
     GraphicsBox[{
       {GrayLevel[0.9], Thickness[0.006], Arrowheads[Medium], 
        ArrowBox[{{0, 0}, {1, 0}}], ArrowBox[{{0, 0}, {0, 1}}]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[{{0, 0}, {1, 0}}]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Cos[10 Degree], Sin[10 Degree]}}, {{0, 0}, {
           0.984807753012208, 0.17364817766693033`}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Cos[20 Degree], Sin[20 Degree]}}, {{0, 0}, {
           0.9396926207859084, 0.3420201433256687}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 2] 3^Rational[1, 2], Rational[1, 2]}}, {{0, 0}, {
           0.8660254037844386, 0.5}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Cos[40 Degree], Sin[40 Degree]}}, {{0, 0}, {
           0.766044443118978, 0.6427876096865393}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Sin[40 Degree], Cos[40 Degree]}}, {{0, 0}, {
           0.6427876096865393, 0.766044443118978}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {{0, 0}, {0.5, 
           0.8660254037844386}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Sin[20 Degree], Cos[20 Degree]}}, {{0, 0}, {
           0.3420201433256687, 0.9396926207859084}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Sin[10 Degree], Cos[10 Degree]}}, {{0, 0}, {
           0.17364817766693033`, 0.984807753012208}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[{{0, 0}, {0, 1}}]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Sin[10 Degree], Cos[10 Degree]}}, {{0, 
           0}, {-0.17364817766693033`, 0.984807753012208}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Sin[20 Degree], Cos[20 Degree]}}, {{0, 
           0}, {-0.3420201433256687, 0.9396926207859084}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[-1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {{0, 0}, {-0.5,
            0.8660254037844386}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Sin[40 Degree], Cos[40 Degree]}}, {{0, 
           0}, {-0.6427876096865393, 0.766044443118978}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Cos[40 Degree], Sin[40 Degree]}}, {{0, 
           0}, {-0.766044443118978, 0.6427876096865393}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[-1, 2] 3^Rational[1, 2], Rational[1, 2]}}, {{0, 
           0}, {-0.8660254037844386, 0.5}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Cos[20 Degree], Sin[20 Degree]}}, {{0, 
           0}, {-0.9396926207859084, 0.3420201433256687}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Cos[10 Degree], Sin[10 Degree]}}, {{0, 
           0}, {-0.984807753012208, 0.17364817766693033`}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[{{0, 0}, {-1, 0}}]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Cos[10 Degree], -Sin[10 Degree]}}, {{0, 
           0}, {-0.984807753012208, -0.17364817766693033`}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Cos[20 Degree], -Sin[20 Degree]}}, {{0, 
           0}, {-0.9396926207859084, -0.3420201433256687}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[-1, 2] 3^Rational[1, 2], Rational[-1, 2]}}, {{0, 
           0}, {-0.8660254037844386, -0.5}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Cos[40 Degree], -Sin[40 Degree]}}, {{0, 
           0}, {-0.766044443118978, -0.6427876096865393}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Sin[40 Degree], -Cos[40 Degree]}}, {{0, 
           0}, {-0.6427876096865393, -0.766044443118978}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[-1, 2], Rational[-1, 2] 3^Rational[1, 2]}}, {{0, 
           0}, {-0.5, -0.8660254037844386}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Sin[20 Degree], -Cos[20 Degree]}}, {{0, 
           0}, {-0.3420201433256687, -0.9396926207859084}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {-Sin[10 Degree], -Cos[10 Degree]}}, {{0, 
           0}, {-0.17364817766693033`, -0.984807753012208}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[{{0, 0}, {0, -1}}]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Sin[10 Degree], -Cos[10 Degree]}}, {{0, 0}, {
           0.17364817766693033`, -0.984807753012208}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Sin[20 Degree], -Cos[20 Degree]}}, {{0, 0}, {
           0.3420201433256687, -0.9396926207859084}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 2], Rational[-1, 2] 3^Rational[1, 2]}}, {{0, 0}, {
           0.5, -0.8660254037844386}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Sin[40 Degree], -Cos[40 Degree]}}, {{0, 0}, {
           0.6427876096865393, -0.766044443118978}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Cos[40 Degree], -Sin[40 Degree]}}, {{0, 0}, {
           0.766044443118978, -0.6427876096865393}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 2] 3^Rational[1, 2], Rational[-1, 2]}}, {{0, 0}, {
           0.8660254037844386, -0.5}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Cos[20 Degree], -Sin[20 Degree]}}, {{0, 0}, {
           0.9396926207859084, -0.3420201433256687}}]]}, 
       {GrayLevel[0.7], Thickness[0.003], Arrowheads[Small], 
        ArrowBox[
         NCache[{{0, 0}, {Cos[10 Degree], -Sin[10 Degree]}}, {{0, 0}, {
           0.984807753012208, -0.17364817766693033`}}]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[ArrowBox[{{0, 0}, {1, 0}}],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[10 Degree]^2, Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
            0.9698463103929541, 0.17101007166283433`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[20 Degree]^2, Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
            0.8830222215594891, 0.3213938048432697}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[3, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {0.75,
             0.4330127018922193}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[40 Degree]^2, Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
            0.5868240888334652, 0.49240387650610395`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[40 Degree]^2, Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
            0.41317591116653474`, 0.49240387650610395`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[1, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {0.25,
             0.4330127018922193}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[20 Degree]^2, Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
            0.11697777844051097`, 0.3213938048432697}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[10 Degree]^2, Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
            0.030153689607045803`, 0.17101007166283433`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[ArrowBox[{{0, 0}, {0, 0}}],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[10 Degree]^2, -Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
            0.030153689607045803`, -0.17101007166283433`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[20 Degree]^2, -Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
            0.11697777844051097`, -0.3213938048432697}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[1, 4], Rational[-1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
            0.25, -0.4330127018922193}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[40 Degree]^2, -Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
            0.41317591116653474`, -0.49240387650610395`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[40 Degree]^2, -Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
            0.5868240888334652, -0.49240387650610395`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[3, 4], Rational[-1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
            0.75, -0.4330127018922193}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[20 Degree]^2, -Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
            0.8830222215594891, -0.3213938048432697}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[10 Degree]^2, -Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
            0.9698463103929541, -0.17101007166283433`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[ArrowBox[{{0, 0}, {1, 0}}],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[10 Degree]^2, Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
            0.9698463103929541, 0.17101007166283433`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[20 Degree]^2, Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
            0.8830222215594891, 0.3213938048432697}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[3, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {0.75,
             0.4330127018922193}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[40 Degree]^2, Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
            0.5868240888334652, 0.49240387650610395`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[40 Degree]^2, Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
            0.41317591116653474`, 0.49240387650610395`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[1, 4], Rational[1, 4] 3^Rational[1, 2]}}, {{0, 0}, {0.25,
             0.4330127018922193}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[20 Degree]^2, Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
            0.11697777844051097`, 0.3213938048432697}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[10 Degree]^2, Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
            0.030153689607045803`, 0.17101007166283433`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[ArrowBox[{{0, 0}, {0, 0}}],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[10 Degree]^2, -Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
            0.030153689607045803`, -0.17101007166283433`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[20 Degree]^2, -Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
            0.11697777844051097`, -0.3213938048432697}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[1, 4], Rational[-1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
            0.25, -0.4330127018922193}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Sin[40 Degree]^2, -Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
            0.41317591116653474`, -0.49240387650610395`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[40 Degree]^2, -Cos[40 Degree] Sin[40 Degree]}}, {{0, 0}, {
            0.5868240888334652, -0.49240387650610395`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Rational[3, 4], Rational[-1, 4] 3^Rational[1, 2]}}, {{0, 0}, {
            0.75, -0.4330127018922193}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[20 Degree]^2, -Cos[20 Degree] Sin[20 Degree]}}, {{0, 0}, {
            0.8830222215594891, -0.3213938048432697}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.], Thickness[0.007], Arrowheads[Medium], 
        StyleBox[
         ArrowBox[
          NCache[{{0, 0}, {
            Cos[10 Degree]^2, -Cos[10 Degree] Sin[10 Degree]}}, {{0, 0}, {
            0.9698463103929541, -0.17101007166283433`}}]],
         FontColor->GrayLevel[0.]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[{{0, 0}, {0, 0}}]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[10 Degree] Sin[10 Degree], Sin[10 Degree]^2}}, {{0, 0}, {
           0.17101007166283433`, 0.030153689607045803`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[20 Degree] Sin[20 Degree], Sin[20 Degree]^2}}, {{0, 0}, {
           0.3213938048432697, 0.11697777844051097`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 0}, {
           0.4330127018922193, 0.25}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[40 Degree] Sin[40 Degree], Sin[40 Degree]^2}}, {{0, 0}, {
           0.49240387650610395`, 0.41317591116653474`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[40 Degree] Sin[40 Degree], Cos[40 Degree]^2}}, {{0, 0}, {
           0.49240387650610395`, 0.5868240888334652}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 4] 3^Rational[1, 2], Rational[3, 4]}}, {{0, 0}, {
           0.4330127018922193, 0.75}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[20 Degree] Sin[20 Degree], Cos[20 Degree]^2}}, {{0, 0}, {
           0.3213938048432697, 0.8830222215594891}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[10 Degree] Sin[10 Degree], Cos[10 Degree]^2}}, {{0, 0}, {
           0.17101007166283433`, 0.9698463103929541}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[{{0, 0}, {0, 1}}]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[10 Degree] Sin[10 Degree], Cos[10 Degree]^2}}, {{0, 
           0}, {-0.17101007166283433`, 0.9698463103929541}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[20 Degree] Sin[20 Degree], Cos[20 Degree]^2}}, {{0, 
           0}, {-0.3213938048432697, 0.8830222215594891}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[-1, 4] 3^Rational[1, 2], Rational[3, 4]}}, {{0, 
           0}, {-0.4330127018922193, 0.75}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[40 Degree] Sin[40 Degree], Cos[40 Degree]^2}}, {{0, 
           0}, {-0.49240387650610395`, 0.5868240888334652}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[40 Degree] Sin[40 Degree], Sin[40 Degree]^2}}, {{0, 
           0}, {-0.49240387650610395`, 0.41317591116653474`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[-1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 
           0}, {-0.4330127018922193, 0.25}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[20 Degree] Sin[20 Degree], Sin[20 Degree]^2}}, {{0, 
           0}, {-0.3213938048432697, 0.11697777844051097`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[10 Degree] Sin[10 Degree], Sin[10 Degree]^2}}, {{0, 
           0}, {-0.17101007166283433`, 0.030153689607045803`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[{{0, 0}, {0, 0}}]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[10 Degree] Sin[10 Degree], Sin[10 Degree]^2}}, {{0, 0}, {
           0.17101007166283433`, 0.030153689607045803`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[20 Degree] Sin[20 Degree], Sin[20 Degree]^2}}, {{0, 0}, {
           0.3213938048432697, 0.11697777844051097`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 0}, {
           0.4330127018922193, 0.25}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[40 Degree] Sin[40 Degree], Sin[40 Degree]^2}}, {{0, 0}, {
           0.49240387650610395`, 0.41317591116653474`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[40 Degree] Sin[40 Degree], Cos[40 Degree]^2}}, {{0, 0}, {
           0.49240387650610395`, 0.5868240888334652}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[1, 4] 3^Rational[1, 2], Rational[3, 4]}}, {{0, 0}, {
           0.4330127018922193, 0.75}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[20 Degree] Sin[20 Degree], Cos[20 Degree]^2}}, {{0, 0}, {
           0.3213938048432697, 0.8830222215594891}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Cos[10 Degree] Sin[10 Degree], Cos[10 Degree]^2}}, {{0, 0}, {
           0.17101007166283433`, 0.9698463103929541}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[{{0, 0}, {0, 1}}]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[10 Degree] Sin[10 Degree], Cos[10 Degree]^2}}, {{0, 
           0}, {-0.17101007166283433`, 0.9698463103929541}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[20 Degree] Sin[20 Degree], Cos[20 Degree]^2}}, {{0, 
           0}, {-0.3213938048432697, 0.8830222215594891}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[-1, 4] 3^Rational[1, 2], Rational[3, 4]}}, {{0, 
           0}, {-0.4330127018922193, 0.75}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[40 Degree] Sin[40 Degree], Cos[40 Degree]^2}}, {{0, 
           0}, {-0.49240387650610395`, 0.5868240888334652}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[40 Degree] Sin[40 Degree], Sin[40 Degree]^2}}, {{0, 
           0}, {-0.49240387650610395`, 0.41317591116653474`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 0}, {
           Rational[-1, 4] 3^Rational[1, 2], Rational[1, 4]}}, {{0, 
           0}, {-0.4330127018922193, 0.25}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[20 Degree] Sin[20 Degree], Sin[20 Degree]^2}}, {{0, 
           0}, {-0.3213938048432697, 0.11697777844051097`}}]]}, 
       {GrayLevel[0.5], Thickness[0.007], Arrowheads[Medium], 
        ArrowBox[
         NCache[{{0, 
           0}, {-Cos[10 Degree] Sin[10 Degree], Sin[10 Degree]^2}}, {{0, 
           0}, {-0.17101007166283433`, 0.030153689607045803`}}]]}}], {576., -189.},
      ImageScaled[{0.5, 0.5}], {360., 360.}]}, {}},
  ContentSelectable->True,
  ImageSize->Full,
  PlotRangePadding->{6, 5}]], "Output",
 CellChangeTimes->{
  3.5591387411938047`*^9, 3.5591388029075127`*^9, 3.5591389561933823`*^9, 
   3.5591390146778846`*^9, 3.559139136607699*^9, 3.559139295936779*^9, 
   3.5591516436684723`*^9, 3.559151683105342*^9, 3.5591517504506598`*^9, {
   3.559151780558713*^9, 3.559151803319153*^9}, 3.559151891194107*^9, 
   3.570135135278653*^9, 3.570135431981574*^9, 3.57013589616039*^9, 
   3.570135929762849*^9, {3.570136002490176*^9, 3.5701360777915087`*^9}, {
   3.570136117930379*^9, 3.5701361322824044`*^9}, {3.570136312244321*^9, 
   3.5701363690908203`*^9}, 3.5701368581360793`*^9, 3.5701369171353827`*^9, 
   3.5701369566502523`*^9, 3.5844640079292507`*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\

Oblique Projection\
\>", "Section",
 CellChangeTimes->{{3.5531876848220286`*^9, 3.553187698362852*^9}, {
  3.5532795512472496`*^9, 3.553279552682452*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{"Choose", " ", "oblique", " ", "angle"}], " ", "*)"}], 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"alpha", " ", "=", " ", "45"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"factor", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"[", 
             RowBox[{"i", ",", "1"}], "]"}], "]"}], "*", 
           RowBox[{"Sin", "[", 
            RowBox[{"alpha", " ", "Degree"}], "]"}]}], " ", "+", " ", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"[", 
             RowBox[{"i", ",", "2"}], "]"}], "]"}], "*", 
           RowBox[{"Cos", "[", 
            RowBox[{"alpha", " ", "Degree"}], "]"}]}]}], ")"}], "/", 
        RowBox[{"Sin", "[", 
         RowBox[{"alpha", " ", "Degree"}], "]"}]}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"aoblique1", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"factor", "[", 
           RowBox[{"[", "i", "]"}], "]"}], "*", 
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "1"}], "]"}], "]"}]}], ",", 
         RowBox[{
          RowBox[{"factor", "[", 
           RowBox[{"[", "i", "]"}], "]"}], "*", 
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "2"}], "]"}], "]"}]}]}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"angles", " ", "wrt", " ", "e2"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"factor2", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"[", 
             RowBox[{"i", ",", "1"}], "]"}], "]"}], "*", 
           RowBox[{"Cos", "[", 
            RowBox[{"alpha", " ", "Degree"}], "]"}]}], " ", "+", " ", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"[", 
             RowBox[{"i", ",", "2"}], "]"}], "]"}], "*", 
           RowBox[{"Sin", "[", 
            RowBox[{"alpha", " ", "Degree"}], "]"}]}]}], ")"}], "/", 
        RowBox[{"Sin", "[", 
         RowBox[{"alpha", " ", "Degree"}], "]"}]}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"aoblique2", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"factor2", "[", 
           RowBox[{"[", "i", "]"}], "]"}], "*", 
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "1"}], "]"}], "]"}]}], ",", 
         RowBox[{
          RowBox[{"factor2", "[", 
           RowBox[{"[", "i", "]"}], "]"}], "*", 
          RowBox[{"u", "[", 
           RowBox[{"[", 
            RowBox[{"i", ",", "2"}], "]"}], "]"}]}]}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Check", " ", "rank", " ", "and", " ", "idempotent", " ", "property", " ",
      "of", " ", "the", " ", "rth", " ", "matrix"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"Do", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{
        RowBox[{"mortho", " ", "=", " ", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"aoblique1", "[", 
            RowBox[{"[", "i", "]"}], "]"}], ",", 
           RowBox[{"aoblique2", "[", 
            RowBox[{"[", "i", "]"}], "]"}]}], "}"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Print", "[", 
          RowBox[{"\"\<Matrix orthogonal = \>\"", ",", 
           RowBox[{
            RowBox[{"MatrixForm", "[", "mortho", "]"}], "//", "N"}]}], "]"}], 
         "\[IndentingNewLine]", 
         RowBox[{"Print", "[", 
          RowBox[{"\"\<  Det = \>\"", ",", 
           RowBox[{"Det", "[", "mortho", "]"}], ",", "\"\<     P*P = \>\"", 
           ",", 
           RowBox[{
            RowBox[{"MatrixForm", "[", 
             RowBox[{"mortho", ".", "mortho"}], "]"}], "//", "N"}]}], 
          "]"}]}]}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}], ";"}], 
    "\[IndentingNewLine]", "*)"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Graphics", " ", "constructions"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"aoblique1arrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{"Thin", ",", " ", "Red", ",", 
          RowBox[{"Arrowheads", "[", "Medium", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"aoblique1", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"aoblique2arrows", " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{"Thin", ",", " ", "Green", ",", 
          RowBox[{"Arrowheads", "[", "Medium", "]"}], ",", 
          RowBox[{"Arrow", "[", 
           RowBox[{"{", 
            RowBox[{"origin", ",", " ", 
             RowBox[{"aoblique2", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"Show", "[", 
    RowBox[{
    "e1e2", ",", " ", "uarrows", ",", " ", "aoblique1arrows", ",", 
     "aoblique2arrows"}], "]"}], " ", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Display", " ", "one", " ", "mapping", " ", "and", " ", "connect", " ", 
     "e1", " ", "to", " ", "a1", " ", "and", " ", "e2", " ", "to", " ", 
     "a2"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"oconnect1", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{"Thin", ",", " ", "Blue", ",", " ", 
          RowBox[{"Line", "[", 
           RowBox[{"{", 
            RowBox[{"e1", ",", 
             RowBox[{"aoblique1", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"oconnect2", " ", "=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{"Thin", ",", " ", "Blue", ",", " ", 
          RowBox[{"Line", "[", 
           RowBox[{"{", 
            RowBox[{"e2", ",", 
             RowBox[{"aoblique2", "[", 
              RowBox[{"[", "i", "]"}], "]"}]}], "}"}], "]"}]}], "}"}], "]"}], 
       ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "numu"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"Manipulate", "[", 
    RowBox[{
     RowBox[{"Show", "[", 
      RowBox[{"e1e2", ",", " ", "uarrows", ",", " ", 
       RowBox[{"thickuarrows", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}], ",", 
       RowBox[{"aoblique1arrows", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}], ",", 
       RowBox[{"aoblique2arrows", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}], ",", " ", 
       RowBox[{"oconnect1", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}], ",", " ", 
       RowBox[{"oconnect2", "[", 
        RowBox[{"[", "uvec", "]"}], "]"}]}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"uvec", ",", " ", 
         RowBox[{"IntegerPart", "[", 
          RowBox[{"numu", "/", "6"}], "]"}]}], "}"}], ",", " ", "1", ",", 
       "numu", ",", "1"}], "}"}], ",", " ", 
     RowBox[{"FrameLabel", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<\>\"", ",", "\"\<\>\"", ",", 
        RowBox[{"alpha", "\"\<Degree Oblique Angle\>\""}], ",", "\"\<\>\""}], 
       "}"}]}]}], "]"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.5531877036356616`*^9, 3.553187944141284*^9}, {
   3.55318804426226*^9, 3.5531881783132954`*^9}, {3.553188223724975*^9, 
   3.5531883272467566`*^9}, 3.553188429161736*^9, {3.5531884829194307`*^9, 
   3.5531884995022593`*^9}, {3.553188552901153*^9, 3.553188555989959*^9}, {
   3.5531886026964407`*^9, 3.5531886077196493`*^9}, {3.5531886762505703`*^9, 
   3.5531886771553717`*^9}, {3.553188718542244*^9, 3.5531887744371424`*^9}, {
   3.553188812470009*^9, 3.553188896039356*^9}, {3.5531889530106564`*^9, 
   3.553188988235518*^9}, {3.553189100524515*^9, 3.5531891098065314`*^9}, {
   3.553189243717167*^9, 3.5531892768360248`*^9}, {3.553189320188501*^9, 
   3.553189322793706*^9}, {3.5531894384211082`*^9, 3.5531894656743565`*^9}, {
   3.5531895266392636`*^9, 3.553189611082212*^9}, {3.553189641159065*^9, 
   3.5531897696252904`*^9}, {3.5531898823510885`*^9, 
   3.5531900709398193`*^9}, {3.5531901249939146`*^9, 
   3.5531901346503315`*^9}, {3.553190203680453*^9, 3.5531902250836906`*^9}, {
   3.5531904033764033`*^9, 3.5531904423140717`*^9}, {3.553191289083559*^9, 
   3.5531913437304554`*^9}, {3.55319464279305*^9, 3.553194643463851*^9}, {
   3.5531946903419333`*^9, 3.5531946916679354`*^9}, {3.5531947241003923`*^9, 
   3.553194744505228*^9}, {3.553194798886924*^9, 3.5531947995265245`*^9}, {
   3.5531948807246675`*^9, 3.5531948808338675`*^9}, {3.5531949393495703`*^9, 
   3.5531949394587708`*^9}, {3.553195111137072*^9, 3.5531951188902855`*^9}, {
   3.5531952547977247`*^9, 3.5531952552189255`*^9}, {3.5531986569221706`*^9, 
   3.5531987398675165`*^9}, 3.5531997367560673`*^9, {3.5532608445350256`*^9, 
   3.5532610234966393`*^9}, {3.5532611727467933`*^9, 
   3.5532611735675945`*^9}, {3.5532612298544874`*^9, 3.553261255776929*^9}, {
   3.553261309826618*^9, 3.553261333571859*^9}, {3.553261401616373*^9, 
   3.5532614145487957`*^9}, {3.5532614612480717`*^9, 
   3.5532615080741453`*^9}, {3.553264538510892*^9, 3.553264553810913*^9}, {
   3.5532727213653374`*^9, 3.5532729011512375`*^9}, {3.55327295386412*^9, 
   3.5532729778341537`*^9}, {3.5532730658519*^9, 3.553273095198347*^9}, {
   3.5532731513880424`*^9, 3.553273163963664*^9}, {3.553273239510191*^9, 
   3.5532732420217953`*^9}, {3.553273491213862*^9, 3.5532734961754704`*^9}, {
   3.5532743351378736`*^9, 3.553274354525104*^9}, {3.553274430061628*^9, 
   3.5532744308728294`*^9}, {3.5532748982173767`*^9, 3.553274916606209*^9}, {
   3.5532750918789062`*^9, 3.5532752441587625`*^9}, {3.5532752926024446`*^9, 
   3.5532752950048485`*^9}, {3.5532755371472635`*^9, 
   3.5532755689856024`*^9}, {3.5532757629922314`*^9, 
   3.5532757631222315`*^9}, {3.553276410752902*^9, 3.553276591516408*^9}, {
   3.5532766340740795`*^9, 3.5532766341988797`*^9}, {3.5532767007901917`*^9, 
   3.553276726895035*^9}, {3.5532768370536184`*^9, 3.553276841759626*^9}, {
   3.553276897942921*^9, 3.553276898052121*^9}, {3.553277076584318*^9, 
   3.553277077383519*^9}, {3.553277340673942*^9, 3.553277385000012*^9}, {
   3.553277514258222*^9, 3.5532775451642747`*^9}, {3.553277577005528*^9, 
   3.55327757800393*^9}, {3.5532776085231805`*^9, 3.553277609318782*^9}, {
   3.5532788745069957`*^9, 3.5532789876991644`*^9}, {3.5532790846621003`*^9, 
   3.55327919164178*^9}, {3.5532792415079412`*^9, 3.5532792965508337`*^9}, {
   3.5532793293368864`*^9, 3.5532793576929326`*^9}, {3.553281652842217*^9, 
   3.553281710924715*^9}, {3.553282549164321*^9, 3.553282567060351*^9}, {
   3.5532873839309015`*^9, 3.553287475382262*^9}, {3.553287505474715*^9, 
   3.5532875636940174`*^9}, {3.553287630072134*^9, 3.553287630103334*^9}, {
   3.5532878034196386`*^9, 3.5532878261332784`*^9}, {3.5532901824741735`*^9, 
   3.553290195624996*^9}, {3.553297503840529*^9, 3.553297537910989*^9}, {
   3.55329758421187*^9, 3.5532975843990707`*^9}, {3.5532976432267737`*^9, 
   3.553297643289174*^9}, 3.553297697327669*^9, {3.5533526591198483`*^9, 
   3.5533526910063043`*^9}, {3.554582323809264*^9, 3.5545823255876675`*^9}, {
   3.5545823573181233`*^9, 3.5545823600793285`*^9}, {3.554582394804989*^9, 
   3.5545823961777916`*^9}, {3.5547357444360523`*^9, 3.5547357445920525`*^9}}],

Cell[BoxData[
 GraphicsBox[{
   {GrayLevel[0], Thickness[0.006], Arrowheads[Large], 
    ArrowBox[{{0, 0}, {1, 0}}], ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[{{0, 0}, {1, 0}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[10 Degree], Sin[10 Degree]}}, {{0, 0}, {
       0.984807753012208, 0.17364817766693033`}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[20 Degree], Sin[20 Degree]}}, {{0, 0}, {
       0.9396926207859084, 0.3420201433256687}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 2] 3^Rational[1, 2], Rational[1, 2]}}, {{0, 0}, {
       0.8660254037844386, 0.5}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[40 Degree], Sin[40 Degree]}}, {{0, 0}, {
       0.766044443118978, 0.6427876096865393}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[40 Degree], Cos[40 Degree]}}, {{0, 0}, {
       0.6427876096865393, 0.766044443118978}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {{0, 0}, {0.5, 
       0.8660254037844386}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[20 Degree], Cos[20 Degree]}}, {{0, 0}, {
       0.3420201433256687, 0.9396926207859084}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[10 Degree], Cos[10 Degree]}}, {{0, 0}, {
       0.17364817766693033`, 0.984807753012208}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[10 Degree], Cos[10 Degree]}}, {{0, 
       0}, {-0.17364817766693033`, 0.984807753012208}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[20 Degree], Cos[20 Degree]}}, {{0, 
       0}, {-0.3420201433256687, 0.9396926207859084}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 2], Rational[1, 2] 3^Rational[1, 2]}}, {{0, 0}, {-0.5, 
       0.8660254037844386}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[40 Degree], Cos[40 Degree]}}, {{0, 
       0}, {-0.6427876096865393, 0.766044443118978}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[40 Degree], Sin[40 Degree]}}, {{0, 
       0}, {-0.766044443118978, 0.6427876096865393}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 2] 3^Rational[1, 2], Rational[1, 2]}}, {{0, 
       0}, {-0.8660254037844386, 0.5}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[20 Degree], Sin[20 Degree]}}, {{0, 
       0}, {-0.9396926207859084, 0.3420201433256687}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[10 Degree], Sin[10 Degree]}}, {{0, 
       0}, {-0.984807753012208, 0.17364817766693033`}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[{{0, 0}, {-1, 0}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[10 Degree], -Sin[10 Degree]}}, {{0, 
       0}, {-0.984807753012208, -0.17364817766693033`}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[20 Degree], -Sin[20 Degree]}}, {{0, 
       0}, {-0.9396926207859084, -0.3420201433256687}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 2] 3^Rational[1, 2], Rational[-1, 2]}}, {{0, 
       0}, {-0.8660254037844386, -0.5}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Cos[40 Degree], -Sin[40 Degree]}}, {{0, 
       0}, {-0.766044443118978, -0.6427876096865393}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[40 Degree], -Cos[40 Degree]}}, {{0, 
       0}, {-0.6427876096865393, -0.766044443118978}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[-1, 2], Rational[-1, 2] 3^Rational[1, 2]}}, {{0, 
       0}, {-0.5, -0.8660254037844386}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[20 Degree], -Cos[20 Degree]}}, {{0, 
       0}, {-0.3420201433256687, -0.9396926207859084}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {-Sin[10 Degree], -Cos[10 Degree]}}, {{0, 
       0}, {-0.17364817766693033`, -0.984807753012208}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[{{0, 0}, {0, -1}}]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[10 Degree], -Cos[10 Degree]}}, {{0, 0}, {
       0.17364817766693033`, -0.984807753012208}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[20 Degree], -Cos[20 Degree]}}, {{0, 0}, {
       0.3420201433256687, -0.9396926207859084}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 2], Rational[-1, 2] 3^Rational[1, 2]}}, {{0, 0}, {
       0.5, -0.8660254037844386}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Sin[40 Degree], -Cos[40 Degree]}}, {{0, 0}, {
       0.6427876096865393, -0.766044443118978}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[
     NCache[{{0, 0}, {Cos[40 Degree], -Sin[40 Degree]}}, {{0, 0}, {
       0.766044443118978, -0.6427876096865393}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[1, 2] 3^Rational[1, 2], Rational[-1, 2]}}, {{0, 0}, {
       0.8660254037844386, -0.5}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[20 Degree], -Sin[20 Degree]}}, {{0, 0}, {
       0.9396926207859084, -0.3420201433256687}}]]}, 
   {RGBColor[1, 1, 0], Thickness[0.003], Arrowheads[Small], 
    ArrowBox[NCache[{{0, 0}, {Cos[10 Degree], -Sin[10 Degree]}}, {{0, 0}, {
       0.984807753012208, -0.17364817766693033`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {1, 0}}]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree]), 2^Rational[1, 2] 
        Sin[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree])}}, {{0, 0}, {1.1408563820557884`, 
       0.20116376126988014`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree]), 2^Rational[1, 2] 
        Sin[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree])}}, {{0, 0}, {1.2044160264027586`, 
       0.4383715832837806}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[3, 2]^Rational[1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2]), 
        2^Rational[-1, 2] (Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {1.1830127018922192`, 
       0.6830127018922192}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[
     NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree]), 2^Rational[1, 2] 
        Sin[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree])}}, {{0, 0}, {1.079227965339569, 
       0.9055797876726387}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree]), 2^Rational[1, 2] 
        Cos[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree])}}, {{0, 0}, {0.9055797876726387, 
       1.079227965339569}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[-1, 2] (Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2]), 
        Rational[3, 2]^Rational[1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {0.6830127018922192, 
       1.1830127018922192`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree]), 2^Rational[1, 2] 
        Cos[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree])}}, {{0, 0}, {0.4383715832837806, 
       1.2044160264027586`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree]), 2^Rational[1, 2] 
        Cos[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree])}}, {{0, 0}, {0.20116376126988014`, 
       1.1408563820557884`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[
     NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree]), 2^Rational[1, 2] 
        Cos[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree])}}, {{0, 0}, {-0.14085638205578854`, 
       0.7988362387301198}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree]), 2^Rational[1, 2] 
        Cos[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree])}}, {{0, 0}, {-0.20441602640275866`, 
       0.5616284167162194}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[-1, 2] (Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2]), 
        Rational[3, 2]^Rational[1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {-0.1830127018922193, 
       0.31698729810778065`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree]), 2^Rational[1, 2] 
        Cos[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree])}}, {{0, 0}, {-0.07922796533956927, 
       0.09442021232736122}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree]), 2^Rational[1, 2] 
        Sin[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree])}}, {{0, 0}, {
       0.09442021232736124, -0.07922796533956927}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Rational[3, 2]^Rational[1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2]), 
        2^Rational[-1, 2] (Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {
       0.31698729810778065`, -0.1830127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree]), 2^Rational[1, 2] 
        Sin[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree])}}, {{0, 0}, {
       0.5616284167162194, -0.2044160264027587}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree]), 2^Rational[1, 2] 
        Sin[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree])}}, {{0, 0}, {
       0.7988362387301198, -0.14085638205578854`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {1, 0}}]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree]), -2^Rational[1, 2] 
        Sin[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree])}}, {{0, 0}, {1.1408563820557884`, 
       0.20116376126988014`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree]), -2^Rational[1, 2] 
        Sin[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree])}}, {{0, 0}, {1.2044160264027586`, 
       0.43837158328378056`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Rational[3, 2]^Rational[1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2]), -2^Rational[-1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {1.1830127018922192`, 
       0.6830127018922192}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree]), -2^Rational[1, 2] 
        Sin[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree])}}, {{0, 0}, {1.079227965339569, 
       0.9055797876726387}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree]), -2^Rational[1, 2] 
        Cos[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree])}}, {{0, 0}, {0.9055797876726387, 
       1.079227965339569}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[-1, 2] (Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2]), -
        Rational[3, 2]^Rational[1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {0.6830127018922192, 
       1.1830127018922192`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree]), -2^Rational[1, 2] 
        Cos[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree])}}, {{0, 0}, {0.43837158328378056`, 
       1.2044160264027586`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree]), -2^Rational[1, 2] 
        Cos[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree])}}, {{0, 0}, {0.20116376126988014`, 
       1.1408563820557884`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree]), -2^Rational[1, 2] 
        Cos[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree])}}, {{0, 0}, {-0.14085638205578854`,
        0.7988362387301198}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree]), -2^Rational[1, 2] 
        Cos[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree])}}, {{0, 0}, {-0.2044160264027587, 
       0.5616284167162194}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[-1, 2] (Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2]), -
        Rational[3, 2]^Rational[1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {-0.1830127018922193, 
       0.31698729810778065`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree]), -2^Rational[1, 2] 
        Cos[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree])}}, {{0, 0}, {-0.07922796533956927, 
       0.09442021232736124}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree]), -2^Rational[1, 2] 
        Sin[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree])}}, {{0, 0}, {
       0.09442021232736122, -0.07922796533956927}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[3, 2]^Rational[1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2]), -2^Rational[-1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {
       0.31698729810778065`, -0.1830127018922193}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree]), -2^Rational[1, 2] 
        Sin[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree])}}, {{0, 0}, {
       0.5616284167162194, -0.20441602640275866`}}]]}, 
   {RGBColor[1, 0, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree]), -2^Rational[1, 2] 
        Sin[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree])}}, {{0, 0}, {
       0.7988362387301198, -0.14085638205578854`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {1, 0}}]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree]), 2^Rational[1, 2] 
        Sin[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree])}}, {{0, 0}, {1.1408563820557884`, 
       0.20116376126988014`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree]), 2^Rational[1, 2] 
        Sin[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree])}}, {{0, 0}, {1.2044160264027586`, 
       0.4383715832837806}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[3, 2]^Rational[1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2]), 
        2^Rational[-1, 2] (Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {1.1830127018922192`, 
       0.6830127018922192}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree]), 2^Rational[1, 2] 
        Sin[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree])}}, {{0, 0}, {1.079227965339569, 
       0.9055797876726387}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree]), 2^Rational[1, 2] 
        Cos[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree])}}, {{0, 0}, {0.9055797876726387, 
       1.079227965339569}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[-1, 2] (Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2]), 
        Rational[3, 2]^Rational[1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {0.6830127018922192, 
       1.1830127018922192`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree]), 2^Rational[1, 2] 
        Cos[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree])}}, {{0, 0}, {0.4383715832837806, 
       1.2044160264027586`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree]), 2^Rational[1, 2] 
        Cos[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree])}}, {{0, 0}, {0.20116376126988014`, 
       1.1408563820557884`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree]), 2^Rational[1, 2] 
        Cos[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree])}}, {{0, 0}, {-0.14085638205578854`, 
       0.7988362387301198}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree]), 2^Rational[1, 2] 
        Cos[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree])}}, {{0, 0}, {-0.20441602640275866`, 
       0.5616284167162194}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[-1, 2] (Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2]), 
        Rational[3, 2]^Rational[1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {-0.1830127018922193, 
       0.31698729810778065`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree]), 2^Rational[1, 2] 
        Cos[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree])}}, {{0, 0}, {-0.07922796533956927, 
       0.09442021232736122}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree]), 2^Rational[1, 2] 
        Sin[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree])}}, {{0, 0}, {
       0.09442021232736124, -0.07922796533956927}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Rational[3, 2]^Rational[1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2]), 
        2^Rational[-1, 2] (Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {
       0.31698729810778065`, -0.1830127018922193}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree]), 2^Rational[1, 2] 
        Sin[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree])}}, {{0, 0}, {
       0.5616284167162194, -0.2044160264027587}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree]), 2^Rational[1, 2] 
        Sin[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree])}}, {{0, 0}, {
       0.7988362387301198, -0.14085638205578854`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {1, 0}}]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree]), -2^Rational[1, 2] 
        Sin[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree])}}, {{0, 0}, {1.1408563820557884`, 
       0.20116376126988014`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree]), -2^Rational[1, 2] 
        Sin[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree])}}, {{0, 0}, {1.2044160264027586`, 
       0.43837158328378056`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-Rational[3, 2]^Rational[1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2]), -2^Rational[-1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {1.1830127018922192`, 
       0.6830127018922192}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Cos[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree]), -2^Rational[1, 2] 
        Sin[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree])}}, {{0, 0}, {1.079227965339569, 
       0.9055797876726387}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree]), -2^Rational[1, 2] 
        Cos[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree])}}, {{0, 0}, {0.9055797876726387, 
       1.079227965339569}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[-1, 2] (Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2]), -
        Rational[3, 2]^Rational[1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {0.6830127018922192, 
       1.1830127018922192`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[
     NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree]), -2^Rational[1, 2] 
        Cos[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree])}}, {{0, 0}, {0.43837158328378056`, 
       1.2044160264027586`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 
       0}, {-2^Rational[1, 2] 
        Sin[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree]), -2^Rational[1, 2] 
        Cos[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree])}}, {{0, 0}, {0.20116376126988014`, 
       1.1408563820557884`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[{{0, 0}, {0, 1}}]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree]), -2^Rational[1, 2] 
        Cos[10 Degree] (-2^Rational[-1, 2] Cos[10 Degree] + 
         2^Rational[-1, 2] Sin[10 Degree])}}, {{0, 0}, {-0.14085638205578854`,
        0.7988362387301198}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree]), -2^Rational[1, 2] 
        Cos[20 Degree] (-2^Rational[-1, 2] Cos[20 Degree] + 
         2^Rational[-1, 2] Sin[20 Degree])}}, {{0, 0}, {-0.2044160264027587, 
       0.5616284167162194}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[-1, 2] (Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2]), -
        Rational[3, 2]^Rational[1, 2] (
         Rational[-1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {-0.1830127018922193, 
       0.31698729810778065`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[
     NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Sin[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree]), -2^Rational[1, 2] 
        Cos[40 Degree] (-2^Rational[-1, 2] Cos[40 Degree] + 
         2^Rational[-1, 2] Sin[40 Degree])}}, {{0, 0}, {-0.07922796533956927, 
       0.09442021232736124}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree]), -2^Rational[1, 2] 
        Sin[40 Degree] (2^Rational[-1, 2] Cos[40 Degree] - 2^Rational[-1, 2] 
         Sin[40 Degree])}}, {{0, 0}, {
       0.09442021232736122, -0.07922796533956927}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       Rational[3, 2]^Rational[1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2]), -2^Rational[-1, 2] (
         Rational[1, 2] Rational[3, 2]^Rational[1, 2] + 
         Rational[-1, 2] 2^Rational[-1, 2])}}, {{0, 0}, {
       0.31698729810778065`, -0.1830127018922193}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree]), -2^Rational[1, 2] 
        Sin[20 Degree] (2^Rational[-1, 2] Cos[20 Degree] - 2^Rational[-1, 2] 
         Sin[20 Degree])}}, {{0, 0}, {
       0.5616284167162194, -0.20441602640275866`}}]]}, 
   {RGBColor[0, 1, 0], Thickness[Tiny], Arrowheads[Medium], 
    ArrowBox[NCache[{{0, 0}, {
       2^Rational[1, 2] 
        Cos[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree]), -2^Rational[1, 2] 
        Sin[10 Degree] (2^Rational[-1, 2] Cos[10 Degree] - 2^Rational[-1, 2] 
         Sin[10 Degree])}}, {{0, 0}, {
       0.7988362387301198, -0.14085638205578854`}}]]}}]], "Output",
 CellChangeTimes->{
  3.55318898935872*^9, 3.5531891105241327`*^9, {3.553189263498001*^9, 
   3.553189277709626*^9}, 3.553189323792107*^9, {3.5531894394507103`*^9, 
   3.553189466563558*^9}, 3.553189509900434*^9, 3.553189540102087*^9, 
   3.5531897056963778`*^9, {3.55318974115524*^9, 3.5531897711228933`*^9}, 
   3.553189940585991*^9, {3.553190048990581*^9, 3.5531900720318213`*^9}, 
   3.553190136038734*^9, {3.553190204897255*^9, 3.5531902261600924`*^9}, {
   3.5531904057320075`*^9, 3.5531904436088743`*^9}, {3.553191290924362*^9, 
   3.5531913445416565`*^9}, 3.5531946448054533`*^9, 3.5531946926195374`*^9, {
   3.5531947277819986`*^9, 3.55319474556603*^9}, 3.5531948004001265`*^9, 
   3.553194881613869*^9, 3.553194940145172*^9, 3.5531951243814955`*^9, 
   3.5531952560769267`*^9, 3.553198743330723*^9, 3.5531991423482237`*^9, 
   3.5531997422004766`*^9, 3.5532608131949725`*^9, 3.553261032660164*^9, 
   3.5532611788076024`*^9, {3.553261232507292*^9, 3.553261269491351*^9}, 
   3.553261336457864*^9, 3.5532615116917515`*^9, 3.5532645594409213`*^9, 
   3.553272984830964*^9, 3.5532730694507055`*^9, 3.5532731871333027`*^9, 
   3.5532734983906736`*^9, 3.5532743802835474`*^9, 3.553274432308032*^9, 
   3.553275252505576*^9, {3.5532765695975704`*^9, 3.553276597148018*^9}, 
   3.553276636195683*^9, {3.553276678216153*^9, 3.5532767277050357`*^9}, 
   3.5532768427112274`*^9, 3.5532768994405236`*^9, 3.5532769503138056`*^9, 
   3.5532770342450495`*^9, 3.5532770785067215`*^9, 3.553277311145894*^9, {
   3.553277344143547*^9, 3.5532773860140133`*^9}, {3.553277516749426*^9, 
   3.5532775467222767`*^9}, 3.5532775800319333`*^9, 3.5532776108319845`*^9, 
   3.553278990569168*^9, {3.553279117222146*^9, 3.5532791606830072`*^9}, 
   3.5532791924648275`*^9, {3.553279245220747*^9, 3.5532792980716352`*^9}, 
   3.553279358792934*^9, 3.553281721106332*^9, 3.553282571487158*^9, 
   3.5532875728980336`*^9, 3.5532876309925356`*^9, {3.5532877775079927`*^9, 
   3.5532878271472797`*^9}, 3.553290199961804*^9, 3.553297545399002*^9, 
   3.5532975885486774`*^9, 3.5532976449271765`*^9, 3.553297698482071*^9, 
   3.5532977996170483`*^9, 3.5533525656048846`*^9, {3.553352741166792*^9, 
   3.553352761214427*^9}, 3.5545817424742436`*^9, 3.554582332233279*^9, 
   3.5545823648529367`*^9, 3.5545824000777984`*^9, 3.55473574903806*^9, 
   3.5591339137297525`*^9, 3.5591387413654046`*^9, 3.559138803032313*^9, 
   3.559138956318182*^9, 3.5591390147870846`*^9, 3.5591391370288997`*^9, 
   3.5591392961863794`*^9, 3.5591516441988735`*^9, 3.559151683479742*^9, 
   3.5591517508250604`*^9, {3.559151780933113*^9, 3.5591518036935534`*^9}, 
   3.559151891584108*^9, 3.570135136074254*^9, 3.5701354322155747`*^9, 
   3.5701369568842525`*^9, 3.5844640085688515`*^9}],

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`uvec$$ = 7, Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`uvec$$], 6}, 1, 36, 1}}, Typeset`size$$ = {
    360., {193., 198.}}, Typeset`update$$ = 0, Typeset`initDone$$, 
    Typeset`skipInitDone$$ = True, $CellContext`uvec$1816$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, "Variables" :> {$CellContext`uvec$$ = 6}, 
      "ControllerVariables" :> {
        Hold[$CellContext`uvec$$, $CellContext`uvec$1816$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> 
      Show[$CellContext`e1e2, $CellContext`uarrows, 
        Part[$CellContext`thickuarrows, $CellContext`uvec$$], 
        Part[$CellContext`aoblique1arrows, $CellContext`uvec$$], 
        Part[$CellContext`aoblique2arrows, $CellContext`uvec$$], 
        Part[$CellContext`oconnect1, $CellContext`uvec$$], 
        Part[$CellContext`oconnect2, $CellContext`uvec$$]], 
      "Specifications" :> {{{$CellContext`uvec$$, 6}, 1, 36, 1}}, 
      "Options" :> {FrameLabel -> {"", "", 45 "Degree Oblique Angle", ""}}, 
      "DefaultOptions" :> {}],
     ImageSizeCache->{425., {257., 262.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellChangeTimes->{
  3.55318898935872*^9, 3.5531891105241327`*^9, {3.553189263498001*^9, 
   3.553189277709626*^9}, 3.553189323792107*^9, {3.5531894394507103`*^9, 
   3.553189466563558*^9}, 3.553189509900434*^9, 3.553189540102087*^9, 
   3.5531897056963778`*^9, {3.55318974115524*^9, 3.5531897711228933`*^9}, 
   3.553189940585991*^9, {3.553190048990581*^9, 3.5531900720318213`*^9}, 
   3.553190136038734*^9, {3.553190204897255*^9, 3.5531902261600924`*^9}, {
   3.5531904057320075`*^9, 3.5531904436088743`*^9}, {3.553191290924362*^9, 
   3.5531913445416565`*^9}, 3.5531946448054533`*^9, 3.5531946926195374`*^9, {
   3.5531947277819986`*^9, 3.55319474556603*^9}, 3.5531948004001265`*^9, 
   3.553194881613869*^9, 3.553194940145172*^9, 3.5531951243814955`*^9, 
   3.5531952560769267`*^9, 3.553198743330723*^9, 3.5531991423482237`*^9, 
   3.5531997422004766`*^9, 3.5532608131949725`*^9, 3.553261032660164*^9, 
   3.5532611788076024`*^9, {3.553261232507292*^9, 3.553261269491351*^9}, 
   3.553261336457864*^9, 3.5532615116917515`*^9, 3.5532645594409213`*^9, 
   3.553272984830964*^9, 3.5532730694507055`*^9, 3.5532731871333027`*^9, 
   3.5532734983906736`*^9, 3.5532743802835474`*^9, 3.553274432308032*^9, 
   3.553275252505576*^9, {3.5532765695975704`*^9, 3.553276597148018*^9}, 
   3.553276636195683*^9, {3.553276678216153*^9, 3.5532767277050357`*^9}, 
   3.5532768427112274`*^9, 3.5532768994405236`*^9, 3.5532769503138056`*^9, 
   3.5532770342450495`*^9, 3.5532770785067215`*^9, 3.553277311145894*^9, {
   3.553277344143547*^9, 3.5532773860140133`*^9}, {3.553277516749426*^9, 
   3.5532775467222767`*^9}, 3.5532775800319333`*^9, 3.5532776108319845`*^9, 
   3.553278990569168*^9, {3.553279117222146*^9, 3.5532791606830072`*^9}, 
   3.5532791924648275`*^9, {3.553279245220747*^9, 3.5532792980716352`*^9}, 
   3.553279358792934*^9, 3.553281721106332*^9, 3.553282571487158*^9, 
   3.5532875728980336`*^9, 3.5532876309925356`*^9, {3.5532877775079927`*^9, 
   3.5532878271472797`*^9}, 3.553290199961804*^9, 3.553297545399002*^9, 
   3.5532975885486774`*^9, 3.5532976449271765`*^9, 3.553297698482071*^9, 
   3.5532977996170483`*^9, 3.5533525656048846`*^9, {3.553352741166792*^9, 
   3.553352761214427*^9}, 3.5545817424742436`*^9, 3.554582332233279*^9, 
   3.5545823648529367`*^9, 3.5545824000777984`*^9, 3.55473574903806*^9, 
   3.5591339137297525`*^9, 3.5591387413654046`*^9, 3.559138803032313*^9, 
   3.559138956318182*^9, 3.5591390147870846`*^9, 3.5591391370288997`*^9, 
   3.5591392961863794`*^9, 3.5591516441988735`*^9, 3.559151683479742*^9, 
   3.5591517508250604`*^9, {3.559151780933113*^9, 3.5591518036935534`*^9}, 
   3.559151891584108*^9, 3.570135136074254*^9, 3.5701354322155747`*^9, 
   3.5701369568842525`*^9, 3.5844640085844517`*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
WindowSize->{1155, 700},
WindowMargins->{{Automatic, 206}, {Automatic, 19}},
FrontEndVersion->"8.0 for Microsoft Windows (64-bit) (October 6, 2011)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[579, 22, 673, 13, 135, "Subtitle"],
Cell[1255, 37, 3081, 55, 884, "Subsubtitle"],
Cell[CellGroupData[{
Cell[4361, 96, 168, 5, 97, "Section"],
Cell[CellGroupData[{
Cell[4554, 105, 17263, 392, 1232, "Input"],
Cell[21820, 499, 23840, 425, 374, "Output"],
Cell[45663, 926, 4419, 73, 474, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[50119, 1004, 9089, 216, 712, "Input"],
Cell[59211, 1222, 28042, 608, 548, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[87302, 1836, 165, 5, 97, "Section"],
Cell[CellGroupData[{
Cell[87492, 1845, 13111, 299, 752, "Input"],
Cell[100606, 2146, 36859, 686, 374, "Output"],
Cell[137468, 2834, 4623, 75, 536, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
