(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     31722,        835]
NotebookOptionsPosition[     30577,        800]
NotebookOutlinePosition[     30998,        816]
CellTagsIndexPosition[     30955,        813]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[TextData[{
 "Mathematical Principles for Scientific Computing and Visualization  \n\
http://www.farinhansford.com/books/scv\nby Gerald Farin and Dianne Hansford\n\
\n",
 StyleBox["Bivariate functions",
  FontSize->16,
  FontColor->RGBColor[0, 1, 0]],
 "\n\nA bivariate function is a function of two (bi-) variables (-variate).\n\
\nLet's look at the first function in this chapter -- f(x,y) = sin(x^2 + \
y^2).\nThe two variables are x and y.\n\nIn the math cell below, you can use \
the left-mouse button to rotate the 3D function.\n(Sometimes these are called \
2-1/2 D functions.)\n\nDo you know how the function is plotted (displayed)?\n\
Evaluation of a function means that we plug in an (x,y) pair and determine \
the function value there.\nMathematica has a default number of evaluations \
that it decides is needed to create a nice looking image. This number will \
vary depending on the complexity of the function.\n\nNotice that when we \
change the domain over which the function is plotted, the plot looks very \
polygonal.\nWhen you dynamically change what needs to be plotted, the only \
way for Mathematica to (re-)evaluate the surface fast enough is to evaluate \
fewer positions."
}], "Input",
 PageWidth->WindowWidth,
 CellChangeTimes->{{3.4160777710925026`*^9, 3.4160777904002657`*^9}, {
   3.4160785180966415`*^9, 3.4160785683188577`*^9}, {3.4160792035022063`*^9, 
   3.4160793390471106`*^9}, 3.416079593733331*^9, {3.4160797209963264`*^9, 
   3.416079820779808*^9}, {3.4160801794555583`*^9, 3.4160802362972927`*^9}, {
   3.416100080762232*^9, 3.416100224559002*^9}, {3.4161003365300083`*^9, 
   3.416100338693118*^9}, {3.4161003729223375`*^9, 3.416100388034067*^9}, {
   3.4270415496055584`*^9, 3.4270415721179295`*^9}},
 Magnification->1.5],

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"sinx2y2", "[", 
     RowBox[{"x_", ",", " ", "y_"}], "]"}], " ", ":=", " ", 
    RowBox[{"Sin", "[", 
     RowBox[{
      RowBox[{"x", "^", "2"}], " ", "+", " ", 
      RowBox[{"y", "^", "2"}]}], "]"}]}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"Manipulate", "[", 
    RowBox[{
     RowBox[{"Plot3D", "[", 
      RowBox[{
       RowBox[{"sinx2y2", "[", 
        RowBox[{"x", ",", "y"}], "]"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"x", ",", " ", 
         RowBox[{"-", "1.5"}], ",", " ", "xmax"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"y", ",", 
         RowBox[{"-", "1.5"}], ",", "ymax"}], "}"}], ",", " ", 
       RowBox[{"AxesLabel", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<x\>\"", ",", "\"\<y\>\"", ",", "\"\<f(x,y)\>\""}], 
         "}"}]}], ",", 
       RowBox[{"ImageSize", "\[Rule]", "400"}], ",", " ", 
       RowBox[{"Mesh", "\[Rule]", "None"}], ",", " ", 
       RowBox[{"PlotPoints", "\[Rule]", "70"}]}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"xmax", ",", " ", "3.14"}], "}"}], ",", " ", "0", ",", "6"}], 
      "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"ymax", ",", " ", "3.14"}], "}"}], ",", " ", "0", ",", "6"}], 
      "}"}]}], "]"}]}]}]], "Input",
 CellChangeTimes->{{3.4160798587243695`*^9, 3.416080079131299*^9}, {
  3.416080147499608*^9, 3.416080155851618*^9}, {3.41615798360428*^9, 
  3.416157985927621*^9}, {3.416158337182701*^9, 3.416158341498907*^9}, {
  3.4166831390736065`*^9, 3.4166831391837645`*^9}, {3.417197994720536*^9, 
  3.4171980211986094`*^9}},
 Magnification->1.5],

Cell["\<\

Here is a simpler function -- that can help us understand bivariate function \
better.

Detect in the plot that the function is quadratic in x and linear in y.

Here we can easily see that Mathematica is plotting univariate functions: \
f(x,constant) and f(constant, y).
The program picks some number of values at which to plot these univariate \
functions that live on the bivariate function.
Can you identify f[x,0] and f[0,y]?\
\>", "Input",
 PageWidth->WindowWidth,
 CellChangeTimes->{{3.4160809325885105`*^9, 3.416081002018346*^9}, {
  3.4160854797670317`*^9, 3.4160856003804655`*^9}},
 Magnification->1.5],

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"fct", "[", 
     RowBox[{"x_", ",", " ", "y_"}], "]"}], " ", ":=", " ", 
    RowBox[{
     RowBox[{"x", "^", "2"}], " ", "+", " ", "y"}]}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"Manipulate", "[", 
    RowBox[{
     RowBox[{"Plot3D", "[", 
      RowBox[{
       RowBox[{"fct", "[", 
        RowBox[{"x", ",", "y"}], "]"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "1"}], ",", " ", "xmax"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"y", ",", 
         RowBox[{"-", "1"}], ",", "ymax"}], "}"}], ",", 
       RowBox[{"PlotRange", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"-", "1"}], ",", "3"}], "}"}]}], "}"}]}], ",", 
       RowBox[{"ImageSize", "\[Rule]", "400"}], ",", " ", 
       RowBox[{"AxesLabel", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<x\>\"", ",", "\"\<y\>\"", ",", "\"\<f(x,y)\>\""}], 
         "}"}]}]}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"xmax", ",", "1"}], "}"}], ",", " ", "0", ",", "1"}], "}"}], 
     ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"ymax", ",", "1"}], "}"}], ",", " ", "0", ",", "1"}], "}"}]}],
     "]"}], "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.4160796259696846`*^9, 3.4160796451072035`*^9}, {
  3.4160802566365395`*^9, 3.416080345223922*^9}, {3.4160804144234257`*^9, 
  3.4160804990951777`*^9}, {3.416080539192835*^9, 3.4160805640585904`*^9}, {
  3.4160805984380255`*^9, 3.416080626608533*^9}, {3.4160807796986656`*^9, 
  3.416080802010749*^9}, {3.4160808495991774`*^9, 3.416080898489478*^9}, {
  3.4161588125562544`*^9, 3.416158813677867*^9}, {3.416683162817749*^9, 
  3.4166831628978643`*^9}},
 Magnification->1.5],

Cell["\<\

A plane can be defined as plane(x,y) = ax + by + c

This is a plane thru the origin with largest slope in the direction (a,b).

Example: plane(x,y) = x+y has largest slope in (1,1)
Example: plane(x,y) = x has largest slope in (1,0)\
\>", "Input",
 CellChangeTimes->{{3.41608142955311*^9, 3.416081475238803*^9}, {
  3.4160851308152637`*^9, 3.4160852076958127`*^9}, {3.416085271387397*^9, 
  3.4160852944405456`*^9}},
 FormatType->"TextForm",
 Magnification->1.5],

Cell[BoxData[{
 RowBox[{
  RowBox[{"a", " ", "=", " ", "1"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"b", " ", "=", " ", "0"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"c", " ", "=", " ", "0"}], ";"}], 
  "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"plane", "[", 
    RowBox[{"x_", ",", "y_"}], "]"}], " ", ":=", " ", 
   RowBox[{
    RowBox[{"a", "*", "x"}], " ", "+", " ", 
    RowBox[{"b", "*", "y"}], " ", "+", " ", "c"}]}], 
  "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
 RowBox[{"Manipulate", "[", 
  RowBox[{
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{"plane", "[", 
      RowBox[{"x", ",", "y"}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", " ", "xmax"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "ymax"}], "}"}], ",", 
     RowBox[{"PlotRange", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "10"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "10"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", 
          RowBox[{"plane", "[", 
           RowBox[{"10", ",", "10"}], "]"}]}], "}"}]}], "}"}]}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "400"}]}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"xmax", ",", "10"}], "}"}], ",", " ", "1", ",", "10"}], "}"}], 
   ",", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"ymax", ",", "10"}], "}"}], ",", " ", "1", ",", "10"}], "}"}]}],
   "]"}], "\[IndentingNewLine]"}], "Input",
 CellChangeTimes->{{3.4160811071695457`*^9, 3.4160811225316353`*^9}, {
  3.416081222535434*^9, 3.416081231788739*^9}, {3.416081406740307*^9, 
  3.4160814266289053`*^9}, {3.4160814818583217`*^9, 3.4160815420148225`*^9}, {
  3.416084155512848*^9, 3.4160842215478015`*^9}, {3.416085011023011*^9, 
  3.4160851074316397`*^9}, {3.416085251418683*^9, 3.416085260621917*^9}, {
  3.416085312947157*^9, 3.416085415144109*^9}, {3.4161592236173315`*^9, 
  3.4161592239678354`*^9}, {3.416683172361472*^9, 3.4166831725217023`*^9}},
 Magnification->1.5],

Cell[TextData[{
 "\n",
 StyleBox["Tangent Plane",
  FontSize->14,
  FontColor->RGBColor[0, 1, 0]],
 "\n\nThe tangent plane to a bivariate function f(x,y) at a particular (x,y) \
is analagous to a tangent to a univariate function at a particular x-value.\n\
\nLet's say we want the tangent plane at (xhat, yhat). To find the tangent \
plane there, we need the concept of a partial derivative of a bivariate \
function.  df(x,yhat)/dx will give us the derivative of the function in x \
that passes thru yhat. df(xhat, y)/dy will give us the derivative of the \
function in y that passes thru xhat. (See the chapter for the correct \
font/notation.)\n\nAs we know from univariate functions, the derivative \
(slope) at a given x-value defines a tangent line there. Thus we use these \
slopes in the x- and y-directions to define the tangent plane.\n\ntangent \
plane at xhat, yhat : \ntangent(x,y) = f(xhat,yhat) + fx(xhat,yhat)(x - xhat) \
+ fy(xhat,yhat)(y - yhat)\nwhere fx and fy are the partial derivatives in x \
and y, respectively.\nSee how similar this is to the tangent to a univariate \
function!"
}], "Input",
 PageWidth->WindowWidth,
 CellChangeTimes->{{3.4160856548287582`*^9, 3.4160856596356707`*^9}, {
  3.4161000215170417`*^9, 3.4161000562169375`*^9}, {3.4161004214220767`*^9, 
  3.416100581632448*^9}, {3.4161006305728207`*^9, 3.4161008936310797`*^9}, {
  3.4161455073667088`*^9, 3.4161455940112977`*^9}},
 Magnification->1.5],

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"Clear", "[", 
     RowBox[{
     "xp", ",", "yp", ",", "myfct", ",", "myfctdx", ",", "myfctdy", ",", " ", 
      "xmin", ",", "xmax", ",", "ymin", ",", "ymax"}], "]"}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Extents", " ", "for", " ", "function", " ", "domain"}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"xmin", " ", "=", " ", 
     RowBox[{"-", "2"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"xmax", " ", "=", " ", "2"}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"ymin", " ", "=", " ", 
     RowBox[{"-", "2"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"ymax", " ", "=", " ", "2"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Define", " ", "function", " ", "and", " ", "its", " ", "partials"}], " ",
     "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"myfct", "[", 
      RowBox[{"x_", ",", " ", "y_"}], "]"}], "=", " ", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"x", "^", "3"}]}], " ", "+", " ", 
      RowBox[{"y", "^", "2"}], " ", "+", " ", "x"}]}], ";"}], " ", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
      RowBox[{"myfct", "[", 
       RowBox[{"x_", ",", " ", "y_"}], "]"}], "=", " ", 
      RowBox[{
       RowBox[{"2", 
        RowBox[{"x", "^", "3"}]}], " ", "-", " ", 
       RowBox[{"y", "^", "2"}], " ", "+", " ", 
       RowBox[{"2", "x"}], " ", "-", " ", "y"}]}], ";"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"myfctdx", "[", 
      RowBox[{"x_", ",", " ", "y_"}], "]"}], "=", " ", 
     RowBox[{"D", "[", 
      RowBox[{
       RowBox[{"myfct", "[", 
        RowBox[{"x", ",", "y"}], "]"}], ",", " ", "x"}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"myfctdy", "[", 
      RowBox[{"x_", ",", " ", "y_"}], "]"}], "=", " ", 
     RowBox[{"D", "[", 
      RowBox[{
       RowBox[{"myfct", "[", 
        RowBox[{"x", ",", "y"}], "]"}], ",", " ", "y"}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"tangentplane", "[", 
     RowBox[{"x_", ",", " ", "y_", ",", " ", "xp_", ",", "yp_"}], "]"}], " ", 
    ":=", " ", 
    RowBox[{
     RowBox[{"myfct", "[", 
      RowBox[{"xp", ",", " ", "yp"}], "]"}], " ", "+", " ", 
     RowBox[{
      RowBox[{"myfctdx", "[", 
       RowBox[{"xp", ",", "yp"}], "]"}], "*", 
      RowBox[{"(", 
       RowBox[{"x", " ", "-", " ", "xp"}], ")"}]}], " ", "+", 
     RowBox[{
      RowBox[{"myfctdy", "[", 
       RowBox[{"yp", ",", " ", "yp"}], "]"}], "*", 
      RowBox[{"(", 
       RowBox[{"y", "-", " ", "yp"}], ")"}]}]}]}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"Manipulate", "[", 
    RowBox[{
     RowBox[{"Plot3D", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"myfct", "[", 
          RowBox[{"x", ",", "y"}], "]"}], ",", " ", 
         RowBox[{"tangentplane", "[", 
          RowBox[{"x", ",", "y", ",", "xpoint", ",", "ypoint"}], "]"}]}], 
        "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"x", ",", "xmin", ",", "xmax"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "ymin", ",", "ymax"}], "}"}], ",", " ", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Directive", "[", 
           RowBox[{"Yellow", ",", 
            RowBox[{"Specularity", "[", 
             RowBox[{"White", ",", "200"}], "]"}], ",", 
            RowBox[{"Opacity", "[", "1.0", "]"}]}], "]"}], ",", "  ", 
          RowBox[{"Directive", "[", 
           RowBox[{"Red", ",", 
            RowBox[{"Specularity", "[", 
             RowBox[{"White", ",", "20"}], "]"}], ",", 
            RowBox[{"Opacity", "[", "0.3", "]"}]}], "]"}]}], "}"}]}], ",", 
       RowBox[{"Mesh", "\[Rule]", "None"}], ",", 
       RowBox[{"PlotRange", "\[Rule]", "Full"}], ",", " ", 
       RowBox[{"AxesLabel", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<x\>\"", ",", "\"\<y\>\"", ",", "\"\<f(x,y)\>\""}], 
         "}"}]}], ",", " ", 
       RowBox[{"ImageSize", "\[Rule]", "400"}]}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"xpoint", ",", "0"}], "}"}], ",", "xmin", ",", "xmax"}], 
      "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"ypoint", ",", "0"}], "}"}], ",", "ymin", ",", "ymax"}], 
      "}"}]}], "]"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.416100898287776*^9, 3.4161010218554573`*^9}, {
   3.4161010613322225`*^9, 3.416101198088869*^9}, {3.416101244335368*^9, 
   3.416101262711792*^9}, {3.4161013420358543`*^9, 3.4161013431374383`*^9}, {
   3.41610144517416*^9, 3.4161014840400467`*^9}, {3.4161016762163825`*^9, 
   3.4161016854196157`*^9}, {3.4161375845142064`*^9, 
   3.4161377296028337`*^9}, {3.416137954606373*^9, 3.416137958341744*^9}, {
   3.4161385287619677`*^9, 3.4161385318764467`*^9}, {3.416141199001584*^9, 
   3.416141255953477*^9}, {3.416141297272891*^9, 3.4161413835870047`*^9}, {
   3.416141424595973*^9, 3.416141446367278*^9}, {3.4161414904807105`*^9, 
   3.416141501166075*^9}, {3.41614161299688*^9, 3.4161416150698605`*^9}, {
   3.416141864658752*^9, 3.416141899088259*^9}, {3.416141958713997*^9, 
   3.4161419880561886`*^9}, {3.4161421244623313`*^9, 
   3.4161421760164623`*^9}, {3.4161424187454896`*^9, 
   3.4161424603453074`*^9}, {3.4161424929521933`*^9, 
   3.4161425425935745`*^9}, {3.416142591584019*^9, 3.416142620315333*^9}, {
   3.4161426842172194`*^9, 3.4161426878925037`*^9}, {3.416142735561048*^9, 
   3.416142736091811*^9}, {3.416142941607328*^9, 3.4161429422983217`*^9}, {
   3.4161430232647457`*^9, 3.4161430275208654`*^9}, {3.416143079235227*^9, 
   3.4161431240096097`*^9}, {3.416143234157995*^9, 3.4161432603656797`*^9}, {
   3.4161433040885506`*^9, 3.4161433705641375`*^9}, {3.4161434101811037`*^9, 
   3.4161434234301553`*^9}, {3.416143463878317*^9, 3.4161434695965395`*^9}, {
   3.416143521591304*^9, 3.4161435394269505`*^9}, {3.4161435767806625`*^9, 
   3.416143588667755*^9}, {3.4161436390502014`*^9, 3.4161436432562494`*^9}, {
   3.4161437053755727`*^9, 3.4161437163914127`*^9}, {3.4161437813448114`*^9, 
   3.416143922487765*^9}, {3.4161439897745185`*^9, 3.416143990986261*^9}, {
   3.416144121654152*^9, 3.416144147541376*^9}, {3.4161442313819327`*^9, 
   3.416144240885598*^9}, {3.41614427933088*^9, 3.4161443012524014`*^9}, {
   3.416144459540008*^9, 3.41614448362464*^9}, {3.4161445244032764`*^9, 
   3.4161445425794125`*^9}, {3.4161446115786285`*^9, 3.416144669211501*^9}, 
   3.416160899617301*^9, {3.416681539984229*^9, 3.4166815400843725`*^9}, {
   3.416683358529168*^9, 3.4166833858584657`*^9}},
 Magnification->1.5],

Cell["\<\

Gradient

The tangent plane at (xp,yp) has the greatest slope in the direction v = \
(fx(xp,yp), fy(xp,yp)).
v is a vector that indicates a direction and speed. v is called the gradient.

If we plot the gradient for the function at some refinement of points on a \
regular grid, it is called a gradient field. The vectors are plotted as \
arrows. The lengths of the arrows indicate the speed -- greater slope will \
require a longer arrow.

Gradient fields are helpful for finding the shortest path to a maximum or \
minimum of a function -- e.g., given today's conditions, what is the fastest \
way to maximum profits!
\
\>", "Input",
 PageWidth->WindowWidth,
 CellChangeTimes->{{3.4161452443785505`*^9, 3.4161452695046797`*^9}, {
  3.4161456026236815`*^9, 3.416145768742549*^9}, {3.4161458140376797`*^9, 
  3.416145866593251*^9}, {3.416146048995533*^9, 3.4161460931189795`*^9}},
 Magnification->1.5],

Cell[BoxData[{
 RowBox[{"Needs", "[", "\"\<VectorFieldPlots`\>\"", 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"Clear", "[", "f", "]"}], ";"}], "\[IndentingNewLine]", 
  "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{"Here", " ", "is", " ", "a", " ", "simple", " ", "function"}], " ",
    "*)"}], "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Zero", " ", "slope", " ", "in", " ", "the", " ", "direction", " ", "of", 
    " ", "the", " ", "gradient", " ", "results", " ", "in", " ", "a", " ", 
    "very", " ", "short", " ", 
    RowBox[{"vector", "."}]}], " ", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"x", "^", "2"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", " ", "2"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", "0", ",", " ", "2"}], "}"}]}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"GradientFieldPlot", "[", 
   RowBox[{
    RowBox[{"x", "^", "2"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", " ", "0", ",", " ", "2"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"y", ",", " ", "0", ",", " ", "2"}], "}"}]}], "]"}], 
  "\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{"This", " ", "is", " ", "a", " ", "little", " ", "more", " ", 
    RowBox[{"complicated", "."}]}], " ", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", " ", "y"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", " ", "2"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", "0", ",", " ", "2"}], "}"}]}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"GradientFieldPlot", "[", 
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{"x", " ", "y"}], "]"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", " ", "0", ",", " ", "2"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"y", ",", " ", "0", ",", " ", "2"}], "}"}]}], "]"}], 
  "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", "x", " ", "]"}], " ", "+", " ", 
    RowBox[{"Cos", "[", "y", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", " ", 
     RowBox[{"4", "Pi"}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", "0", ",", " ", 
     RowBox[{"4", "Pi"}]}], "}"}]}], "]"}], "\[IndentingNewLine]", 
 RowBox[{"GradientFieldPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", "x", " ", "]"}], " ", "+", " ", 
    RowBox[{"Cos", "[", "y", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", 
     RowBox[{"4", "Pi"}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", "0", ",", " ", 
     RowBox[{"4", "Pi"}]}], "}"}]}], "]"}]}], "Input",
 CellChangeTimes->{{3.4161458723014593`*^9, 3.41614601723987*^9}, {
   3.4161461017614064`*^9, 3.416146189808011*^9}, {3.4161462536998835`*^9, 
   3.416146278585667*^9}, {3.416146610613099*^9, 3.416146611203949*^9}, 
   3.4161473668505154`*^9},
 Magnification->1.5],

Cell["\<\

Contours

A contour of a bivariate function is the set of points where f(x,y) = \
constant.
For example, we might want to know where the function is zero: f(x,y) = 0.
As the default setting, Mathematica creates contours for several different \
\"constant\" values. Color shades are used to accent the regions in between \
the \"constant\" value contour curves. These colors tell us that all values \
of the function in this region are bounded by the contour around it.
\
\>", "Input",
 PageWidth->WindowWidth,
 CellChangeTimes->{{3.4161472805364017`*^9, 3.41614732971712*^9}, {
  3.4161478095771246`*^9, 3.416147959522736*^9}},
 Magnification->1.5],

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Cos", "[", "x", " ", "]"}], " ", "+", " ", 
      RowBox[{"Cos", "[", "y", "]"}]}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", " ", "0", ",", " ", 
       RowBox[{"4", "Pi"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"y", ",", " ", "0", ",", " ", 
       RowBox[{"4", "Pi"}]}], "}"}]}], "]"}], "\[IndentingNewLine]", 
   RowBox[{"ContourPlot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Cos", "[", "x", "]"}], "+", 
      RowBox[{"Cos", "[", "y", "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", 
       RowBox[{"4", "Pi"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", 
       RowBox[{"4", "Pi"}]}], "}"}]}], "]"}]}]}]], "Input",
 CellChangeTimes->{{3.4161473384396625`*^9, 3.416147360711688*^9}},
 Magnification->1.5],

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "^", "2"}], "-", 
      RowBox[{"x", "^", "2"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "4"}], ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "4"}], ",", "4"}], "}"}], ",", 
     RowBox[{"MeshFunctions", "->", 
      RowBox[{"{", 
       RowBox[{"#3", "&"}], "}"}]}], ",", 
     RowBox[{"Mesh", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"{", "0", "}"}], "}"}]}], ",", 
     RowBox[{"BoxRatios", "->", 
      RowBox[{"{", 
       RowBox[{"1", ",", "1", ",", "1"}], "}"}]}]}], "]"}], 
   "\[IndentingNewLine]", 
   RowBox[{"ContourPlot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "^", "2"}], "-", 
      RowBox[{"x", "^", "2"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "4"}], ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "4"}], ",", "4"}], "}"}]}], "]"}]}]}]], "Input",
 CellChangeTimes->{{3.4161477311443443`*^9, 3.416147746586549*^9}},
 Magnification->1.5],

Cell["\<\

Here is another application of the contour function -- finding the \
intersection of two functions.\
\>", "Input",
 PageWidth->WindowWidth,
 CellChangeTimes->{{3.4161479758962803`*^9, 3.416147993101019*^9}},
 Magnification->1.5],

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", 
        RowBox[{"y", "^", "2"}]}], ",", 
       RowBox[{"x", "+", "y", " ", "-", "0.2"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", " ", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}], "\[IndentingNewLine]", 
   
   RowBox[{"ContourPlot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], " ", "==", " ", 
      RowBox[{"x", "+", "y", " ", "-", "0.2"}]}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", " ", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}]}]}]], "Input",
 CellChangeTimes->{{3.4161474207580304`*^9, 3.4161474369112577`*^9}, {
  3.4161475020549297`*^9, 3.416147629047536*^9}, {3.41616139325712*^9, 
  3.416161451320611*^9}},
 Magnification->1.5],

Cell["\<\

Let's sneak in a trivariate function ... that is a function of three \
variables. So for each position in (x,y,z) space we have an associated \
function value -- for instance pressure or temperature.x^2, y^2

f(x,y,z) = x^2 + y^2 + z^2

Can you make sense of what you see here?!\
\>", "Input",
 CellChangeTimes->{{3.416148241097621*^9, 3.4161482941739407`*^9}, {
  3.416148394948848*^9, 3.4161484370393715`*^9}, {3.4161485080014095`*^9, 
  3.416148516373448*^9}},
 EmphasizeSyntaxErrors->True,
 Magnification->1.5],

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"ContourPlot3D", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], " ", "+", " ", 
      RowBox[{"y", "^", "2"}], " ", "+", " ", 
      RowBox[{"z", "^", "2"}]}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", " ", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"y", ",", " ", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"z", ",", " ", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}], 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.4161483864466224`*^9, 3.416148391193448*^9}, {
  3.416148440584469*^9, 3.416148484998333*^9}, {3.416668474857499*^9, 
  3.4166684830993505`*^9}},
 Magnification->1.5],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"SetDirectory", "[", 
     RowBox[{"ToFileName", "[", 
      RowBox[{"Extract", "[", 
       RowBox[{
        RowBox[{"\"\<FileName\>\"", "/.", "\[InvisibleSpace]", 
         RowBox[{"NotebookInformation", "[", 
          RowBox[{"EvaluationNotebook", "[", "]"}], "]"}]}], ",", 
        RowBox[{"{", "1", "}"}], ",", "FrontEnd`FileName"}], "]"}], "]"}], 
     "]"}], ";"}], ")"}], " ", 
  "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"x", "^", " ", 
    RowBox[{"+", " ", 
     RowBox[{"y", "^", "2"}]}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", " ", "2"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", "0", ",", " ", "2"}], "}"}]}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{"gradpic", " ", "=", " ", 
  RowBox[{"GradientFieldPlot", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "2"}], " ", "+", " ", 
     RowBox[{"y", "^", "2"}]}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", " ", "0", ",", " ", "2"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"y", ",", " ", "0", ",", " ", "2"}], "}"}]}], 
   "]"}]}], "\[IndentingNewLine]", 
 RowBox[{"Export", "[", 
  RowBox[{"\"\<gradientfield1.eps\>\"", ",", " ", "gradpic"}], 
  "]"}]}], "Input",
 CellChangeTimes->{{3.416665133723184*^9, 3.416665162584685*^9}, {
  3.4166652996217346`*^9, 3.416665341552027*^9}, {3.4166654857694016`*^9, 
  3.416665486650669*^9}, {3.4166664556239843`*^9, 3.4166664574866624`*^9}, {
  3.416667429804787*^9, 3.4166674394186115`*^9}, {3.4198840482283263`*^9, 
  3.4198840499508033`*^9}}],

Cell[BoxData["\<\"gradientfield1.eps\"\>"], "Output",
 CellChangeTimes->{{3.416665145660349*^9, 3.4166651682828784`*^9}, 
   3.416665204955611*^9, 3.416665490686472*^9, 3.4166664620732574`*^9, 
   3.416667440139648*^9, 3.419884061417291*^9, 3.427041663729661*^9}]
}, Open  ]],

Cell[BoxData[{
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"SetDirectory", "[", 
     RowBox[{"ToFileName", "[", 
      RowBox[{"Extract", "[", 
       RowBox[{
        RowBox[{"\"\<FileName\>\"", "/.", "\[InvisibleSpace]", 
         RowBox[{"NotebookInformation", "[", 
          RowBox[{"EvaluationNotebook", "[", "]"}], "]"}]}], ",", 
        RowBox[{"{", "1", "}"}], ",", "FrontEnd`FileName"}], "]"}], "]"}], 
     "]"}], ";"}], ")"}], " ", 
  "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"x", "^", "2"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", " ", "2"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", "0", ",", "2"}], "}"}]}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{"c1", " ", "=", " ", 
  RowBox[{"ContourPlot", "[", 
   RowBox[{
    RowBox[{"x", "^", "2"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "2"}], "}"}]}], 
   "]"}]}], "\[IndentingNewLine]", 
 RowBox[{"Export", "[", 
  RowBox[{"\"\<contour.jpeg\>\"", ",", " ", "c1"}], "]"}]}], "Input",
 CellChangeTimes->{{3.4166680033294754`*^9, 3.4166680293468866`*^9}, {
  3.4166680606418867`*^9, 3.4166681110844193`*^9}}]
},
WindowSize->{1014, 698},
WindowMargins->{{118, Automatic}, {Automatic, 16}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 1767, 29, 649, "Input"],
Cell[2338, 52, 1749, 45, 194, "Input"],
Cell[4090, 99, 622, 16, 283, "Input"],
Cell[4715, 117, 2114, 55, 221, "Input"],
Cell[6832, 174, 472, 13, 193, "Input"],
Cell[7307, 189, 2162, 56, 302, "Input"],
Cell[9472, 247, 1441, 25, 533, "Input"],
Cell[10916, 274, 6956, 160, 761, "Input"],
Cell[17875, 436, 912, 22, 418, "Input"],
Cell[18790, 460, 3089, 81, 437, "Input"],
Cell[21882, 543, 659, 16, 283, "Input"],
Cell[22544, 561, 916, 26, 113, "Input"],
Cell[23463, 589, 1153, 36, 140, "Input"],
Cell[24619, 627, 239, 7, 103, "Input"],
Cell[24861, 636, 1162, 34, 113, "Input"],
Cell[26026, 672, 525, 14, 171, "Input"],
Cell[26554, 688, 802, 22, 113, "Input"],
Cell[CellGroupData[{
Cell[27381, 714, 1644, 42, 159, "Input"],
Cell[29028, 758, 263, 3, 48, "Output"]
}, Open  ]],
Cell[29306, 764, 1267, 34, 159, "Input"]
}
]
*)

(* End of internal cache information *)
