(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     63659,       1242]
NotebookOptionsPosition[     63088,       1220]
NotebookOutlinePosition[     63506,       1236]
CellTagsIndexPosition[     63463,       1233]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell["\<\

Mathematical Principles for Scientific Computing and Visualization  
http://www.farinhansford.com/books/scv
by Gerald Farin and Dianne Hansford

This notebook demonstrates an application of discrete partial differential \
equations. 

Here we take a rectangular patch with polygonal  boundaries and minimize \
Laplace's equation, f_xx + f_yy = 0, to find the interior points. The number \
of iterations done to minimize Laplace's equation determines the quality of \
the solution.

Elements to play with:
1)  The number of points on the boundary -- the discretization of the \
polynomials that define the boundaries. (This is \"n\" below.)
2) The number of iterations. In the first cell, you can manually set \
\"numiter\"; in the manipulate cell below, Mathematica will animate this \
variable.
3) Try different boundary curves -- fo, f1, g0, g1

In the first cell below, you change the values manually.
In the second cell below, you can manipulate the \"numiter\" variable.

Figure 11.11 was created with this notebook.

See Chapter 11: Computing with Multivariate Functionsfor details.

Updated May 2008\
\>", "Input",
 PageWidth->WindowWidth,
 CellChangeTimes->{{3.4206546585790157`*^9, 3.420654672428931*^9}, 
   3.420655636184744*^9, {3.420657151864184*^9, 3.4206571555194397`*^9}, {
   3.4206646870404415`*^9, 3.420665005678621*^9}, {3.4263633667926064`*^9, 
   3.4263634350808*^9}, {3.4263659107606525`*^9, 3.4263659207950816`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Set", " ", "the", " ", "output", " ", "folder", " ", "to", " ", "be", " ",
     "the", " ", "one", " ", "this", " ", "file", " ", "lives", " ", "in"}], 
   " ", "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"SetDirectory", "[", 
      RowBox[{"ToFileName", "[", 
       RowBox[{"Extract", "[", 
        RowBox[{
         RowBox[{"\"\<FileName\>\"", "/.", "\[InvisibleSpace]", 
          RowBox[{"NotebookInformation", "[", 
           RowBox[{"EvaluationNotebook", "[", "]"}], "]"}]}], ",", 
         RowBox[{"{", "1", "}"}], ",", "FrontEnd`FileName"}], "]"}], "]"}], 
      "]"}], ";"}], ")"}], " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   
   RowBox[{"(*", " ", 
    RowBox[{
    "Define", " ", "the", " ", "four", " ", "boundary", " ", "curves"}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"f0", " ", "and", " ", "f1", " ", "are", " ", "opposing"}], ";", 
     " ", 
     RowBox[{"g0", " ", "and", " ", "g1", " ", "are", " ", "opposing"}]}], 
    " ", "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"f0", "[", "x_", "]"}], ":=", 
      RowBox[{"8.`", " ", "x", " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", "x"}], ")"}]}]}], ";"}], ")"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"f1", "[", "x_", "]"}], ":=", 
      RowBox[{"8.`", " ", "x", " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", "x"}], ")"}]}]}], ";"}], ")"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"g0", "[", "y_", "]"}], ":=", 
      RowBox[{"8.`", " ", "y", " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", "y"}], ")"}]}]}], ";"}], ")"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"g1", "[", "y_", "]"}], ":=", 
      RowBox[{"8.`", " ", "y", " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", "y"}], ")"}]}]}], ";"}], ")"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Define", " ", "the", " ", "discretization", " ", "of", " ", "the", " ", 
     "boundary", " ", "curves"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"n", "=", "10"}], ";"}], ")"}], " ", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"**", "**", "**", "**", "**"}], "*"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "How", " ", "many", " ", "iterations", " ", "will", " ", "we", " ", "do", 
     " ", "to", " ", "satisfy", " ", "the", " ", "equation"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"numiter", "=", "50"}], ";"}], ")"}], " ", "\[IndentingNewLine]",
    
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"**", "**", "**", "**", "**"}], "*"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Initialize", " ", "the", " ", "nxnx", " ", "table", " ", "with", " ", 
     "zeros"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"F", "=", 
      RowBox[{"Table", "[", 
       RowBox[{"0.`", ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", "1", ",", "n"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"j", ",", "1", ",", "n"}], "}"}]}], "]"}]}], ";"}], ")"}], 
   " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
     "Set", " ", "up", " ", "a", " ", "simple", " ", "name", " ", "for", " ", 
      "n"}], "-", "1"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"n1", "=", 
      RowBox[{"n", "-", "1"}]}], ";"}], ")"}], " ", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Discretize", " ", "the", " ", "boundary", " ", "curves", " ", "and", " ",
      "load", " ", "table", " ", "F", " ", "with", " ", "values", " ", "at", 
     " ", "n", " ", "places"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Do", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"F", "\[LeftDoubleBracket]", 
        RowBox[{"1", ",", "i"}], "\[RightDoubleBracket]"}], "=", 
       RowBox[{"f0", "[", 
        FractionBox[
         RowBox[{"i", "-", "1"}], "n1"], "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], " ", ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"F", "\[LeftDoubleBracket]", 
         RowBox[{"n", ",", "i"}], "\[RightDoubleBracket]"}], "=", 
        RowBox[{"f1", "[", 
         FractionBox[
          RowBox[{"i", "-", "1"}], "n1"], "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], ";"}], ")"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"F", "\[LeftDoubleBracket]", 
         RowBox[{"i", ",", "1"}], "\[RightDoubleBracket]"}], "=", 
        RowBox[{"g0", "[", 
         FractionBox[
          RowBox[{"i", "-", "1"}], "n1"], "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], ";"}], ")"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"F", "\[LeftDoubleBracket]", 
         RowBox[{"i", ",", "n"}], "\[RightDoubleBracket]"}], "=", 
        RowBox[{"g1", "[", 
         FractionBox[
          RowBox[{"i", "-", "1"}], "n1"], "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], ";"}], ")"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"G", " ", "=", " ", "F"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"F", "\[LeftDoubleBracket]", 
     RowBox[{"1", ",", "2"}], "\[RightDoubleBracket]"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Update", " ", "the", " ", "mask", " ", "numiter", " ", "times"}],
     " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
      RowBox[{"The", " ", "mask", " ", "is", " ", "minimizing", " ", 
       RowBox[{"Laplace", "'"}], "s", " ", "equation", " ", "fxx"}], " ", "+",
       " ", "fyy"}], " ", "=", " ", "0"}], " ", "*)"}], "\[IndentingNewLine]",
    "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{"Do", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"F", "\[LeftDoubleBracket]", 
           RowBox[{"i", ",", "j"}], "\[RightDoubleBracket]"}], "=", 
          RowBox[{
           FractionBox["1", "4.`"], 
           RowBox[{"(", 
            RowBox[{
             RowBox[{"2", " ", 
              RowBox[{"F", "\[LeftDoubleBracket]", 
               RowBox[{
                RowBox[{"i", "+", "1"}], ",", "j"}], 
               "\[RightDoubleBracket]"}]}], "+", 
             RowBox[{"2", " ", 
              RowBox[{"F", "\[LeftDoubleBracket]", 
               RowBox[{
                RowBox[{"i", "-", "1"}], ",", "j"}], 
               "\[RightDoubleBracket]"}]}], "+", 
             RowBox[{"2", " ", 
              RowBox[{"F", "\[LeftDoubleBracket]", 
               RowBox[{"i", ",", 
                RowBox[{"j", "-", "1"}]}], "\[RightDoubleBracket]"}]}], "+", 
             RowBox[{"2", " ", 
              RowBox[{"F", "\[LeftDoubleBracket]", 
               RowBox[{"i", ",", 
                RowBox[{"j", "+", "1"}]}], "\[RightDoubleBracket]"}]}], "-", 
             RowBox[{"F", "\[LeftDoubleBracket]", 
              RowBox[{
               RowBox[{"i", "-", "1"}], ",", 
               RowBox[{"j", "-", "1"}]}], "\[RightDoubleBracket]"}], "-", 
             RowBox[{"F", "\[LeftDoubleBracket]", 
              RowBox[{
               RowBox[{"i", "+", "1"}], ",", 
               RowBox[{"j", "-", "1"}]}], "\[RightDoubleBracket]"}], "-", 
             RowBox[{"F", "\[LeftDoubleBracket]", 
              RowBox[{
               RowBox[{"i", "+", "1"}], ",", 
               RowBox[{"j", "+", "1"}]}], "\[RightDoubleBracket]"}], "-", 
             RowBox[{"F", "\[LeftDoubleBracket]", 
              RowBox[{
               RowBox[{"i", "-", "1"}], ",", 
               RowBox[{"j", "+", "1"}]}], "\[RightDoubleBracket]"}]}], 
            ")"}]}]}], ",", 
         RowBox[{"{", 
          RowBox[{"i", ",", "2", ",", 
           RowBox[{"n", "-", "1"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"j", ",", "2", ",", 
           RowBox[{"n", "-", "1"}]}], "}"}]}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"k", ",", "1", ",", "numiter"}], "}"}]}], "]"}], ";"}], ")"}],
    "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", " ", 
   RowBox[{"(*", " ", 
    RowBox[{"F", "\[LeftDoubleBracket]", 
     RowBox[{"1", ",", "2"}], "\[RightDoubleBracket]"}], "  ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Create", " ", "a", " ", "plot", " ", "of", " ", "the", " ", 
     RowBox[{"table", " ", "--"}], " ", "function"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "make", " ", "the", " ", "surface", " ", "a", " ", "bit", " ", "opaque"}],
     " ", "*)"}], " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"picfct", "=", 
     RowBox[{"ListPlot3D", "[", 
      RowBox[{"F", ",", 
       RowBox[{"Mesh", "\[Rule]", "False"}], ",", 
       RowBox[{"Axes", "\[Rule]", "False"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", " ", 
   RowBox[{
    RowBox[{"pic0", "=", " ", 
     RowBox[{"ListPlot3D", "[", 
      RowBox[{"G", ",", " ", 
       RowBox[{"Mesh", "\[Rule]", "True"}], ",", 
       RowBox[{"Axes", "\[Rule]", "False"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{"Show", "[", 
    RowBox[{"picfct", ",", " ", "pic0"}], "]"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Output", " ", "the", " ", "postscript", " ", "file"}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "THis", " ", "takes", " ", "more", " ", "time", " ", "for", " ", "the", 
     " ", "nb", " ", "to", " ", 
     RowBox[{"run", ".", " ", "Uncomment"}], " ", "when", " ", "wanted"}], 
    " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Export", "[", 
       RowBox[{"\"\<Coons.eps\>\"", ",", "pic"}], "]"}], ";"}], ")"}], " ", 
    "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.420655675010573*^9, 3.420655732603387*^9}, {
   3.420655858824885*^9, 3.4206558897593665`*^9}, {3.420655935465088*^9, 
   3.4206559761035233`*^9}, {3.4206568101127696`*^9, 
   3.4206569445060177`*^9}, {3.420656979967008*^9, 3.420657101421651*^9}, {
   3.420657166264891*^9, 3.4206572407620125`*^9}, {3.4206573141174927`*^9, 
   3.420657317312086*^9}, {3.420657356198002*^9, 3.420657444484952*^9}, {
   3.42066210568864*^9, 3.4206621124083023`*^9}, {3.4206621766106205`*^9, 
   3.4206622035894146`*^9}, {3.420662335218688*^9, 3.4206623353288465`*^9}, {
   3.4206625209657793`*^9, 3.4206625232891197`*^9}, 3.4206636527832527`*^9, {
   3.4206650284513664`*^9, 3.4206650595060205`*^9}, {3.420665116487957*^9, 
   3.420665153691453*^9}, {3.4206652305419583`*^9, 3.420665327170904*^9}, {
   3.420665411221763*^9, 3.420665426183277*^9}, {3.4206655144602127`*^9, 
   3.420665534088437*^9}, {3.4263631788323326`*^9, 3.426363192501989*^9}, {
   3.426363297432872*^9, 3.426363327285798*^9}}],

Cell[BoxData[
 Graphics3DBox[{GraphicsComplex3DBox[CompressedData["
1:eJx11gtoU1ccBvDLKnJLxQZ7kWt3WW8v2XRQ7TYc1qH007U+1mI3MVZBO+em
i0zXylRGi/XBpNVI3WbF0jra4KOIxUdbsU5Wl9lH6FZ81KVJTenTQWV1d/gA
IahReg/kIzkkHPLd3HPOL/kn56RuLFy56Q1Jkr4KP+Mkq5mZkb1osPLBMV/o
4rKxieuyyE9NWdBfnfJ0IreJXOvcHn6EJnJF5Gte3/BiIldj5FqMcfQY8xox
1mkXeSRLvEbk+y2vhK5EV+Wzoz7ySkBJ/Pj7J/3kleAMD7Sxso+8EjYsLomv
SwiSV0Ly979e/uRckLwSvNu1sndFrot8OKPixuzdAfJKWOVwtN/MDZKXXaZw
RH5ullfGpw6H46K3h7wyGj/MTMu60EFeGa++l8HN18gr43Dlvvrs0kbyytDj
8uYV/9tAXhlVtwr2r3c1kVfGqC+0t+r8JfLKKI4fTyno6iUvu0yx3sj6sbw2
TMq78k7nf3+T14ZZcXn5u4Nt5LVhhTN9UeHkJvLasL87MSGppJa84fsy01Y+
bMgnrw2fVYxUdaceIK8NwRznzHTp9PVIb3j8+ntJNW/5ycsuU6wr8ndkeRWM
ZlSsLXf3kldBWXZXm2dBO3kV/L/jm287rjaSV8FIjjNl6G4deRV4327+eteB
DeRVXtV//oysg+RVkNVfPPTLFy3kVXArMe9EkRQgL7tMMX90r4o2Y9P8FYf9
5FVR1J3Y6vySvSr6x3wDa2vYq2JX95GFj86wV0XdP2Vvnk8vJ6+KqpGW5A7P
EfKqeLrDldO0+jp5VTy+UZpwrIe97DLFPNHrWcMfPQvHMsrZq2FO3IUf9l1m
r4a630LP3TLXs4aHIenBtDNczxrW/dRYc7PpEHk1/K4tda//bg95NbznTl26
88E18moY3nLI7tnWR152mWK86P9XOraW+pfPXcb1rKM8+6PNAU8neXWsmtzq
GVhylbw6vKEfR9NdDeTVcbe69q/pS+rJq2PSePv9wsdnyasDtwuMY0Wl5NVR
nbL6ydwQe9lleY0Y+5GB5uNd7Un3A+Q1kOs83ue+EyCvAf+f3nP+Qd6PDLTs
HGq1m/fIa+Bnd21uci/vRwbShqfEf/A570cGpradXj7D1SvWbfVagu/Rx80D
5GWXSddFg5VHP2/YEf28YUf084Yd0c8bnGsxxtFFHv28weu0I7KXpJcnWeRH

    "], {{
      {EdgeForm[None], 
       GraphicsGroup3DBox[
        Polygon3DBox[{{12, 1, 2}, {15, 4, 5}, {13, 2, 3}, {20, 9, 10}, {14, 3,
          4}, {19, 8, 9}, {17, 6, 7}, {16, 5, 6}, {18, 7, 8}, {22, 11, 12}, {
         30, 19, 20}, {24, 13, 14}, {54, 43, 44}, {23, 12, 13}, {26, 15, 
         16}, {28, 17, 18}, {29, 18, 19}, {27, 16, 17}, {25, 14, 15}, {32, 21,
          22}, {34, 23, 24}, {38, 27, 28}, {36, 25, 26}, {35, 24, 25}, {37, 
         26, 27}, {43, 32, 33}, {40, 29, 30}, {39, 28, 29}, {33, 22, 23}, {44,
          33, 34}, {53, 42, 43}, {49, 38, 39}, {47, 36, 37}, {46, 35, 36}, {
         48, 37, 38}, {45, 34, 35}, {52, 41, 42}, {50, 39, 40}, {88, 99, 
         98}, {55, 44, 45}, {57, 46, 47}, {60, 49, 50}, {58, 47, 48}, {66, 55,
          56}, {59, 48, 49}, {65, 54, 55}, {63, 52, 53}, {62, 51, 52}, {64, 
         53, 54}, {67, 56, 57}, {76, 65, 66}, {69, 58, 59}, {56, 45, 46}, {68,
          57, 58}, {72, 61, 62}, {74, 63, 64}, {75, 64, 65}, {73, 62, 63}, {
         77, 66, 67}, {78, 67, 68}, {79, 68, 69}, {82, 71, 72}, {83, 72, 
         73}, {84, 73, 74}, {85, 74, 75}, {86, 75, 76}, {87, 76, 77}, {88, 77,
          78}, {80, 69, 70}, {89, 78, 79}, {92, 81, 82}, {95, 84, 85}, {93, 
         82, 83}, {94, 83, 84}, {96, 85, 86}, {90, 79, 80}, {42, 31, 32}, {98,
          87, 88}, {97, 86, 87}, {99, 88, 89}, {100, 89, 90}, {1, 12, 11}, {5,
          16, 15}, {3, 14, 13}, {22, 33, 32}, {4, 15, 14}, {9, 20, 19}, {7, 
         18, 17}, {6, 17, 16}, {8, 19, 18}, {11, 22, 21}, {19, 30, 29}, {16, 
         27, 26}, {14, 25, 24}, {13, 24, 23}, {15, 26, 25}, {12, 23, 22}, {18,
          29, 28}, {17, 28, 27}, {21, 32, 31}, {23, 34, 33}, {43, 54, 53}, {
         25, 36, 35}, {26, 37, 36}, {27, 38, 37}, {28, 39, 38}, {29, 40, 
         39}, {31, 42, 41}, {32, 43, 42}, {24, 35, 34}, {33, 44, 43}, {35, 46,
          45}, {38, 49, 48}, {36, 47, 46}, {37, 48, 47}, {39, 50, 49}, {34, 
         45, 44}, {44, 55, 54}, {42, 53, 52}, {41, 52, 51}, {70, 59, 60}, {45,
          56, 55}, {76, 87, 86}, {55, 66, 65}, {49, 60, 59}, {56, 67, 66}, {
         47, 58, 57}, {51, 62, 61}, {54, 65, 64}, {48, 59, 58}, {52, 63, 
         62}, {57, 68, 67}, {64, 75, 74}, {59, 70, 69}, {58, 69, 68}, {65, 76,
          75}, {2, 13, 12}, {62, 73, 72}, {63, 74, 73}, {61, 72, 71}, {53, 64,
          63}, {66, 77, 76}, {67, 78, 77}, {73, 84, 83}, {71, 82, 81}, {69, 
         80, 79}, {72, 83, 82}, {77, 88, 87}, {75, 86, 85}, {74, 85, 84}, {78,
          89, 88}, {81, 92, 91}, {87, 98, 97}, {68, 79, 78}, {84, 95, 94}, {
         82, 93, 92}, {83, 94, 93}, {85, 96, 95}, {46, 57, 56}, {86, 97, 
         96}, {79, 90, 89}, {89, 100, 99}}]]}, {}, {}, {}}, 
     Line3DBox[{2, 1, 11, 21, 31, 41, 51, 61, 71, 81, 91, 92, 93, 94, 95, 96, 
      97, 98, 99, 100, 90, 80, 70, 60, 50, 40, 30, 20, 10, 9, 8, 7, 6, 5, 4, 
      3, 2}]},
    VertexNormals->CompressedData["
1:eJw9lWs41Osaxp2yKIdURtIqpS4VJgzJiAdpk3IsUayoSMSq5JRJJMo4NO2i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     "]], GraphicsComplex3DBox[CompressedData["
1:eJx9m0+IZFcVxsuZtKnUtDVV3TWV6urXU69evQiuBFcKQl+yEnTlShCkDQiC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    "], {{
      {EdgeForm[None], GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJwtmHkUVlUVxd87l0kRmWSSeZ5E5kmQQUAmGY1BZAZRREFRU5xySMkQ0hxQ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          "]], Polygon3DBox[CompressedData["
1:eJwtmHkcl1MWxu89v9apCS2iRRslLZL2LC0aayESQjEfw/QhWxODYZBhjBki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          "]], Polygon3DBox[CompressedData["
1:eJwtl2W4VVUQhs9ZBy7dHdId0t3ddekOle7uUFI6RemS7pDuFkW6RTqkpBvf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          "]]}]}, {}, {}, {}}, {
      Line3DBox[{101, 1, 387, 11, 398, 399, 21, 418, 419, 31, 438, 439, 41, 
       457, 51, 468, 469, 61, 488, 489, 71, 508, 509, 81, 527, 91, 236, 92, 
       237, 238, 93, 239, 240, 94, 241, 242, 95, 243, 96, 244, 245, 97, 246, 
       247, 98, 248, 249, 99, 250, 100, 535, 90, 525, 524, 80, 505, 504, 70, 
       485, 484, 60, 465, 50, 455, 454, 40, 435, 434, 30, 415, 414, 20, 395, 
       10, 129, 9, 126, 125, 8, 122, 121, 7, 118, 117, 6, 115, 5, 112, 111, 4,
        108, 107, 3, 104, 103, 2, 101}], {
       Line3DBox[{387, 251, 386, 656, 572, 761, 388, 657, 573, 762, 389, 658, 
        574, 763, 390, 659, 536, 391, 660, 575, 764, 392, 661, 576, 765, 393, 
        662, 577, 766, 394, 663, 537, 395}], 
       Line3DBox[{399, 664, 538, 397, 665, 268, 401, 666, 580, 771, 403, 667, 
        582, 774, 405, 668, 539, 407, 670, 671, 540, 409, 672, 586, 779, 411, 
        673, 588, 782, 413, 674, 541, 415}], 
       Line3DBox[{414, 783, 589, 412, 781, 780, 587, 410, 778, 777, 585, 408, 
        776, 584, 669, 406, 775, 583, 404, 773, 772, 581, 402, 770, 769, 579, 
        400, 768, 267, 396, 767, 578, 398}], 
       Line3DBox[{419, 675, 542, 417, 677, 678, 543, 421, 679, 285, 423, 680, 
        593, 789, 425, 681, 544, 427, 683, 684, 545, 429, 686, 687, 546, 431, 
        688, 598, 795, 433, 689, 547, 435}], 
       Line3DBox[{434, 796, 599, 432, 794, 793, 597, 430, 792, 596, 685, 428, 
        791, 595, 682, 426, 790, 594, 424, 788, 787, 592, 422, 786, 284, 420, 
        785, 591, 676, 416, 784, 590, 418}], 
       Line3DBox[{439, 690, 548, 437, 692, 693, 549, 441, 695, 696, 550, 443, 
        697, 302, 445, 698, 551, 447, 700, 701, 552, 449, 703, 704, 553, 451, 
        706, 707, 554, 453, 708, 555, 455}], 
       Line3DBox[{454, 805, 607, 452, 804, 606, 705, 450, 803, 605, 702, 448, 
        802, 604, 699, 446, 801, 603, 444, 800, 301, 442, 799, 602, 694, 440, 
        798, 601, 691, 436, 797, 600, 438}], 
       Line3DBox[{457, 608, 806, 456, 709, 609, 807, 458, 710, 610, 808, 459, 
        711, 611, 809, 460, 318, 461, 712, 612, 810, 462, 713, 613, 811, 463, 
        714, 614, 812, 464, 715, 556, 465}], 
       Line3DBox[{469, 716, 557, 467, 717, 617, 816, 471, 718, 619, 819, 473, 
        719, 621, 822, 475, 720, 558, 477, 721, 335, 479, 722, 624, 827, 481, 
        723, 626, 830, 483, 724, 559, 485}], 
       Line3DBox[{484, 831, 627, 482, 829, 828, 625, 480, 826, 825, 623, 478, 
        824, 334, 476, 823, 622, 474, 821, 820, 620, 472, 818, 817, 618, 470, 
        815, 814, 616, 466, 813, 615, 468}], 
       Line3DBox[{489, 725, 560, 487, 727, 728, 561, 491, 729, 631, 836, 493, 
        730, 633, 839, 495, 731, 562, 497, 733, 734, 563, 499, 735, 352, 501, 
        736, 637, 845, 503, 737, 564, 505}], 
       Line3DBox[{504, 846, 638, 502, 844, 843, 636, 500, 842, 351, 498, 841, 
        635, 732, 496, 840, 634, 494, 838, 837, 632, 492, 835, 834, 630, 490, 
        833, 629, 726, 486, 832, 628, 488}], 
       Line3DBox[{509, 738, 565, 507, 740, 741, 566, 511, 743, 744, 567, 513, 
        745, 643, 852, 515, 746, 568, 517, 748, 749, 569, 519, 751, 752, 570, 
        521, 753, 369, 523, 754, 571, 525}], 
       Line3DBox[{524, 857, 647, 522, 856, 368, 520, 855, 646, 750, 518, 854, 
        645, 747, 516, 853, 644, 514, 851, 850, 642, 512, 849, 641, 742, 510, 
        848, 640, 739, 506, 847, 639, 508}], 
       Line3DBox[{527, 648, 858, 526, 755, 649, 859, 528, 756, 650, 860, 529, 
        757, 651, 861, 530, 652, 862, 531, 758, 653, 863, 532, 759, 654, 864, 
        533, 760, 655, 865, 534, 385, 535}]}, {
       Line3DBox[{101, 251, 102, 767, 266, 664, 131, 784, 281, 675, 146, 797, 
        296, 690, 161, 806, 311, 176, 813, 326, 716, 191, 832, 341, 725, 206, 
        847, 356, 738, 221, 858, 371, 236}], 
       Line3DBox[{103, 252, 656, 105, 267, 665, 132, 282, 676, 677, 147, 297, 
        691, 692, 162, 312, 709, 177, 814, 327, 717, 192, 342, 726, 727, 207, 
        357, 739, 740, 222, 372, 755, 237}], 
       Line3DBox[{107, 254, 657, 109, 769, 269, 666, 134, 284, 679, 149, 299, 
        694, 695, 164, 314, 710, 179, 817, 329, 718, 194, 834, 344, 729, 209, 
        359, 742, 743, 224, 374, 756, 239}], 
       Line3DBox[{111, 256, 658, 113, 772, 271, 667, 136, 787, 286, 680, 151, 
        301, 697, 166, 316, 711, 181, 820, 331, 719, 196, 837, 346, 730, 211, 
        850, 361, 745, 226, 376, 757, 241}], 
       Line3DBox[{115, 258, 659, 116, 775, 273, 668, 138, 790, 288, 681, 153, 
        801, 303, 698, 168, 318, 183, 823, 333, 720, 198, 840, 348, 731, 213, 
        853, 363, 746, 228, 862, 378, 243}], 
       Line3DBox[{117, 259, 660, 119, 274, 669, 670, 139, 289, 682, 683, 154, 
        304, 699, 700, 169, 319, 712, 184, 334, 721, 199, 349, 732, 733, 214, 
        364, 747, 748, 229, 379, 758, 244}], 
       Line3DBox[{121, 261, 661, 123, 777, 276, 672, 141, 291, 685, 686, 156, 
        306, 702, 703, 171, 321, 713, 186, 825, 336, 722, 201, 351, 735, 216, 
        366, 750, 751, 231, 381, 759, 246}], 
       Line3DBox[{125, 263, 662, 127, 780, 278, 673, 143, 793, 293, 688, 158, 
        308, 705, 706, 173, 323, 714, 188, 828, 338, 723, 203, 843, 353, 736, 
        218, 368, 753, 233, 383, 760, 248}], 
       Line3DBox[{129, 265, 663, 130, 783, 280, 674, 145, 796, 295, 689, 160, 
        805, 310, 708, 175, 325, 715, 190, 831, 340, 724, 205, 846, 355, 737, 
        220, 857, 370, 754, 235, 385, 250}], 
       Line3DBox[{238, 373, 859, 223, 741, 358, 848, 208, 728, 343, 833, 193, 
        328, 816, 815, 178, 313, 807, 163, 693, 298, 798, 148, 678, 283, 785, 
        133, 268, 768, 106, 253, 761, 104}], 
       Line3DBox[{240, 375, 860, 225, 744, 360, 849, 210, 345, 836, 835, 195, 
        330, 819, 818, 180, 315, 808, 165, 696, 300, 799, 150, 285, 786, 135, 
        270, 771, 770, 110, 255, 762, 108}], 
       Line3DBox[{242, 377, 861, 227, 362, 852, 851, 212, 347, 839, 838, 197, 
        332, 822, 821, 182, 317, 809, 167, 302, 800, 152, 287, 789, 788, 137, 
        272, 774, 773, 114, 257, 763, 112}], 
       Line3DBox[{245, 380, 863, 230, 749, 365, 854, 215, 734, 350, 841, 200, 
        335, 824, 185, 320, 810, 170, 701, 305, 802, 155, 684, 290, 791, 140, 
        671, 275, 776, 120, 260, 764, 118}], 
       Line3DBox[{247, 382, 864, 232, 752, 367, 855, 217, 352, 842, 202, 337, 
        827, 826, 187, 322, 811, 172, 704, 307, 803, 157, 687, 292, 792, 142, 
        277, 779, 778, 124, 262, 765, 122}], 
       Line3DBox[{249, 384, 865, 234, 369, 856, 219, 354, 845, 844, 204, 339, 
        830, 829, 189, 324, 812, 174, 707, 309, 804, 159, 294, 795, 794, 144, 
        279, 782, 781, 128, 264, 766, 126}]}, {}, {}}},
    VertexNormals->CompressedData["
1:eJztewk0lW3btpJoEJVMKZWhMkaSoTo1CskQheqRmWaNolA0qigJlRQiEsq8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     "]]},
  BoxRatios->{1, 1, 0.4},
  Method->{"RotationControl" -> "Globe"},
  PlotRange->{{1., 10.}, {1., 10.}, {-0.12241669267992336`, 
   1.9753086419753085`}},
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{3.426363169298624*^9, 3.4263632991253057`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
    RowBox[{"**", "**", "**", "\[IndentingNewLine]", "Now"}], " ", 
    RowBox[{"let", "'"}], "s", " ", "use", " ", "the", " ", "manipuate", " ", 
    "command", " ", "to", " ", "change", " ", "the", " ", "number", " ", "of",
     " ", 
    RowBox[{"iterations", " ", "--"}], " ", 
    RowBox[{"numiter", " ", "\[IndentingNewLine]", "**", "**", "**"}]}], " ", 
   "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"createF", "[", "numI_", "]"}], " ", ":=", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"F", "=", 
       RowBox[{"Table", "[", 
        RowBox[{"0.`", ",", 
         RowBox[{"{", 
          RowBox[{"i", ",", "1", ",", "n"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"j", ",", "1", ",", "n"}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Do", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"F", "\[LeftDoubleBracket]", 
          RowBox[{"1", ",", "i"}], "\[RightDoubleBracket]"}], "=", 
         RowBox[{"f0", "[", 
          FractionBox[
           RowBox[{"i", "-", "1"}], "n1"], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], " ", ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Do", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"F", "\[LeftDoubleBracket]", 
          RowBox[{"n", ",", "i"}], "\[RightDoubleBracket]"}], "=", 
         RowBox[{"f1", "[", 
          FractionBox[
           RowBox[{"i", "-", "1"}], "n1"], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Do", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"F", "\[LeftDoubleBracket]", 
          RowBox[{"i", ",", "1"}], "\[RightDoubleBracket]"}], "=", 
         RowBox[{"g0", "[", 
          FractionBox[
           RowBox[{"i", "-", "1"}], "n1"], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], ";", 
      "\[IndentingNewLine]", " ", 
      RowBox[{"Do", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"F", "\[LeftDoubleBracket]", 
          RowBox[{"i", ",", "n"}], "\[RightDoubleBracket]"}], "=", 
         RowBox[{"g1", "[", 
          FractionBox[
           RowBox[{"i", "-", "1"}], "n1"], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Do", "[", 
       RowBox[{
        RowBox[{"Do", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"F", "\[LeftDoubleBracket]", 
            RowBox[{"i", ",", "j"}], "\[RightDoubleBracket]"}], "=", 
           RowBox[{
            FractionBox["1", "4.`"], 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"2", " ", 
               RowBox[{"F", "\[LeftDoubleBracket]", 
                RowBox[{
                 RowBox[{"i", "+", "1"}], ",", "j"}], 
                "\[RightDoubleBracket]"}]}], "+", 
              RowBox[{"2", " ", 
               RowBox[{"F", "\[LeftDoubleBracket]", 
                RowBox[{
                 RowBox[{"i", "-", "1"}], ",", "j"}], 
                "\[RightDoubleBracket]"}]}], "+", 
              RowBox[{"2", " ", 
               RowBox[{"F", "\[LeftDoubleBracket]", 
                RowBox[{"i", ",", 
                 RowBox[{"j", "-", "1"}]}], "\[RightDoubleBracket]"}]}], "+", 
              
              RowBox[{"2", " ", 
               RowBox[{"F", "\[LeftDoubleBracket]", 
                RowBox[{"i", ",", 
                 RowBox[{"j", "+", "1"}]}], "\[RightDoubleBracket]"}]}], "-", 
              
              RowBox[{"F", "\[LeftDoubleBracket]", 
               RowBox[{
                RowBox[{"i", "-", "1"}], ",", 
                RowBox[{"j", "-", "1"}]}], "\[RightDoubleBracket]"}], "-", 
              RowBox[{"F", "\[LeftDoubleBracket]", 
               RowBox[{
                RowBox[{"i", "+", "1"}], ",", 
                RowBox[{"j", "-", "1"}]}], "\[RightDoubleBracket]"}], "-", 
              RowBox[{"F", "\[LeftDoubleBracket]", 
               RowBox[{
                RowBox[{"i", "+", "1"}], ",", 
                RowBox[{"j", "+", "1"}]}], "\[RightDoubleBracket]"}], "-", 
              RowBox[{"F", "\[LeftDoubleBracket]", 
               RowBox[{
                RowBox[{"i", "-", "1"}], ",", 
                RowBox[{"j", "+", "1"}]}], "\[RightDoubleBracket]"}]}], 
             ")"}]}]}], ",", 
          RowBox[{"{", 
           RowBox[{"i", ",", "2", ",", 
            RowBox[{"n", "-", "1"}]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"j", ",", "2", ",", 
            RowBox[{"n", "-", "1"}]}], "}"}]}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"k", ",", "1", ",", "numI"}], "}"}]}], "]"}], ";", " ", 
      RowBox[{"Return", "[", 
       RowBox[{"ListPlot3D", "[", 
        RowBox[{"F", ",", 
         RowBox[{"Mesh", "\[Rule]", "False"}], ",", 
         RowBox[{"Axes", "\[Rule]", "False"}], ",", " ", 
         RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], "]"}], ";"}], 
     ")"}]}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"Manipulate", "[", 
    RowBox[{
     RowBox[{"createF", "[", "numI", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"numI", ",", "0"}], "}"}], ",", " ", "0", ",", "400", ",", 
       " ", "5"}], "}"}]}], "]"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.4206636277873106`*^9, 3.4206636406958723`*^9}, {
  3.4206636759766035`*^9, 3.4206636844287567`*^9}, {3.4206637258683443`*^9, 
  3.420663768880192*^9}, {3.4206641625162125`*^9, 3.4206642169745197`*^9}, {
  3.4206644888454514`*^9, 3.4206645000215216`*^9}, {3.426363442521499*^9, 
  3.42636347254467*^9}}],

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`numI$$ = 75, Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`numI$$], 0}, 0, 400, 5}}, Typeset`size$$ = {
    450., {179., 186.}}, Typeset`update$$ = 0, Typeset`initDone$$, 
    Typeset`skipInitDone$$ = True, $CellContext`numI$2409$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, "Variables" :> {$CellContext`numI$$ = 0}, 
      "ControllerVariables" :> {
        Hold[$CellContext`numI$$, $CellContext`numI$2409$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, 
      "Body" :> $CellContext`createF[$CellContext`numI$$], 
      "Specifications" :> {{{$CellContext`numI$$, 0}, 0, 400, 5}}, 
      "Options" :> {}, "DefaultOptions" :> {}],
     ImageSizeCache->{499., {229.5, 236.5}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellChangeTimes->{{3.420664140574662*^9, 3.4206642368731327`*^9}, {
   3.4206644140879555`*^9, 3.420664429199685*^9}, {3.4206645017740417`*^9, 
   3.4206645167155266`*^9}, 3.4263632011344013`*^9}]
}, Open  ]]
},
WindowSize->{745, 826},
WindowMargins->{{49, Automatic}, {Automatic, 0}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 1451, 35, 596, "Input"],
Cell[CellGroupData[{
Cell[2044, 60, 12191, 300, 1893, "Input"],
Cell[14238, 362, 41198, 669, 396, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[55473, 1036, 5890, 145, 698, "Input"],
Cell[61366, 1183, 1706, 34, 500, "Output"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
