(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     77455,       1541]
NotebookOptionsPosition[     76490,       1506]
NotebookOutlinePosition[     76910,       1522]
CellTagsIndexPosition[     76867,       1519]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[TextData[{
 "\nMathematical Principles for Scientific Computing and Visualization  \n\
http://www.farinhansford.com/books/scv\nby Gerald Farin and Dianne Hansford\n\
\n\n\n",
 StyleBox["Derivatives",
  FontSize->16],
 "\n\nDerivatives help us understand the behaviour of functions.\nDerivatives \
are functions themselves.\n\nSuppose f[t] = 40t^2 represents milage driven \
over time t (hours).\n\nThe derivative of f, called fderiv[t] below, \
represents the instantaneous velocity at time t when f represents distance.  \
Sometimes the derivative is written as f'(t). Let h be an interval of time.\n\
\nFor the function here: f'(t) = lim as h->0 (40(t+h)^2 - 40t^2)/h = 80t \
miles/hour (mph). Notice the units.\n\nThe velocity for our given function is \
linear over time -- our speed is increasing over time. (This isn't a very \
realistic model -- after 5 hours we are going 400 mph!) \n\nThe second \
derivative is the derivative of the first derivative, and it represents \
acceleration when f' represents velocity; thus it is the rate of change of \
velocity.\nf''(t) = (f'(t))' = lim as t->0 of (80(t+h) - 80t) / h = 80 \
miles/hour^2\nTherefore, in this example, acceleration is constant.\n"
}], "Input",
 CellChangeTimes->{{3.4154788699872303`*^9, 3.415478875515179*^9}, {
   3.4154789447146835`*^9, 3.415479009888398*^9}, {3.4154790455396624`*^9, 
   3.4154793091787567`*^9}, {3.4154793530218*^9, 3.415479486483709*^9}, {
   3.4154796224992895`*^9, 3.4154797424517727`*^9}, {3.4154797988829165`*^9, 
   3.4154798054022913`*^9}, {3.4154798428561473`*^9, 
   3.4154798650981293`*^9}, {3.415480140474101*^9, 3.41548020474652*^9}, {
   3.4154802832794447`*^9, 3.415480306222435*^9}, {3.4154803388793936`*^9, 
   3.415480344317213*^9}, {3.415480493541787*^9, 3.415480495634797*^9}, {
   3.4154821925147905`*^9, 3.4154822054834385`*^9}, 3.427042285854232*^9},
 Magnification->1.5],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"Needs", "[", "\"\<PlotLegends`\>\"", "]"}], 
  "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"f", "[", "t_", "]"}], " ", ":=", " ", 
  RowBox[{"40", 
   RowBox[{"t", "^", "2"}]}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"fderiv", "[", "t_", "]"}], " ", ":=", " ", 
  RowBox[{"80", "*", "t"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"fsecderiv", "[", "t_", "]"}], " ", ":=", " ", "80"}], 
  "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"f", "[", "t", "]"}], ",", " ", 
     RowBox[{"fderiv", "[", "t", "]"}], ",", " ", 
     RowBox[{"fsecderiv", "[", "t", "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "5"}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Thickness", "[", "0.005", "]"}], ",", " ", 
      RowBox[{"Thickness", "[", "0.002", "]"}], ",", 
      RowBox[{"Thickness", "[", "0.002", "]"}]}], "}"}]}], ",", " ", 
   RowBox[{"PlotLegend", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Style", "[", 
       RowBox[{"\"\<f\>\"", ",", " ", "15", ",", "  ", 
        RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}], ",", 
      " ", 
      RowBox[{"Style", "[", 
       RowBox[{"\"\<f'\>\"", ",", " ", "15", ",", "  ", 
        RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}], ",", 
      " ", 
      RowBox[{"Style", "[", 
       RowBox[{"\"\<f''\>\"", ",", " ", "15", ",", "  ", 
        RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}]}], 
     "}"}]}], ",", 
   RowBox[{"LegendPosition", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"0.99", ",", " ", 
      RowBox[{"-", "0.3"}]}], "}"}]}]}], " ", "]"}]}], "Input",
 CellChangeTimes->{{3.415478648628933*^9, 3.415478684350298*^9}, {
   3.415478762132142*^9, 3.415478853623701*^9}, {3.415478932547187*^9, 
   3.415478935641637*^9}, {3.415479021094512*^9, 3.415479041443773*^9}, {
   3.415481106743528*^9, 3.415481107494608*^9}, 3.415481336924512*^9},
 Magnification->1.5],

Cell[BoxData[
 GraphicsBox[{InsetBox[
    GraphicsBox[{{}, {}, 
      {Hue[0.67, 0.6, 0.6], Thickness[0.005], LineBox[CompressedData["
1:eJwV02k41AsbBnBrWSLbcVCi8w7Kmjci4XnehkSSypatUIa2U0q2iizJoUaI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        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.002], 
       LineBox[CompressedData["
1:eJwV0n84E3gcB3ALU36UTVfndLacS1t+zHXXb30/pfwIO5WFOY5hm5kfG2Po
h1BuXSLnijqWk7q6rFO5x1yE/Fx+FF1X6CGFKzzJ2ikz7uuPz/N5Xn+838/z
eZ7PWk7cgcglenp6TDyLu/goh+gckbbTJldXVmL9YmdTL9XbnxqIqpbSKDH8
e2ihpS6TSo1C96SKy+v53ai4PodiSE1BTQwRj8gfREIXScwMRYZaT/d4aHhv
Ua+WXzBFKUJt2zKVE7wFZNQ/bDdJuYbaz/aL/uWtgFKVXDROUSLPWiPmKI8C
ipuyfqGjCnWabVpmEMwAHnpUNCHuRbty99Zf/XYLxI7QXt5RPEORTrNmzmYA
xmuz7Aa3DCFvWuS7Ve574Iz5A3pp6Et06/ERSfsWD+D0+93T7xhF28j0mwbH
vEHUnMWJs3uDeNr84vIAX6iMNz34qeckimsZ23jT5wCUpkW/YfhPIRmb+KVO
6AcZ8rn97yfeIU5rGteZfwj0eofgbqoa1RtZ7o8ODIB1iVypH12DGKT0c68T
2fBKc/6QSe1/KFjtYs8Ufgf7RgRH8kUf0C8lzWtmckKA+Spvg7X1LKo8s62/
50AovLnPSW+r0CJxjYtvsXcYOJKefO76vQ7VqsqnZUEcMEaasAH1POK6b62N
CggH90n2jA1LD4p+c66KFkTA08YIAtueAGfUtcfjkyOh6oemEKqGAMPXJUl8
MRe4hpKhrq4lUO+WX5iRwYNwcVqV6QV9mGP2Deem88GJt6HLO8wAtPAn4VJK
FHST7U0kWw0hc/WMT0+2AP5SZFy01yeCgB3eOJMZDWkEum/HcyL0Hf86Av0s
BOOBVI/YO0Zg+lNRn6w0BkaurC2QH14KJeb0lNaLsVCw3tJAyVoGSjFlr93v
cZB3gdsaRjWGC9RLa1z84oFdENxlOmYM1Ydn9fgP4+HDnvKP8joTsPIymvvC
XwQvR3ZLG7NNQdFN+/HZYxH86uJs1RlmBida5hnUYDFUnJCQeA7L4XTCa0Ho
gBiyWeFe+prlkPiJ9QqbwAT4ZqD0WnYn/jOaoSpxKAEKaRcrOr3MYV5NsGwI
SYRj9TmFXkpzkDMqj5aMJYKPFatcZUsCKuc6UxcjgV2xrUEkGQkeeSpvscYl
IF/Dse3QkKC1LNUBJSSB7f2r55hBZJi8vX+HzXQS5Dk00nvqyPC+Wlx1MjYZ
uGqDR+10CwCG297DmmTgVE5TZbkWUBPsFBWdIIU21dLMu3MWkKpTlY1rpSB0
fZHXju1eYKnRYpvfVZY8wV65ge9mOieFgApBzRS2IsDwtT32WG7HO1udBby4
jRxjsYmsvNAcbA/BneopbNfB1TtD5nH+n+JOjU4Ko/5TXtHYQzET1sR5KZx6
2BYoxb5hsD1+FXZPQ4okH9vtq6fkzdicy/03WrBTc8gBUuz0KLmV0wLOu2YP
z2LbDifTdmBX9P290WQB38P23eyJnSqyPWGFbe5NOBiOvVLeYLcDW+4Yfuo8
tvusVngEe/fV7YWXscln99XlYI9SVl6pXOy3K1pRgu1Iam54sNjH2nSrDrtH
Vtz9FNttIku/GztpSdLzkcX+zF6/QezP0pjj04t5S5srb7Hr1Os+LmAr/oif
WcD+HzJXRyk=
        "]]}, 
      {Hue[0.14213595499957954`, 0.6, 0.6], Thickness[0.002], 
       LineBox[{{1.0204081632653061`*^-7, 80.}, {0.04907170402423416, 80.}, {
        0.09814330600765199, 80.}, {0.20454178542729323`, 80.}, {
        0.30388941785076046`, 80.}, {0.40128824409438024`, 80.}, {
        0.5069423457908055, 80.}, {0.6055456004910569, 80.}, {
        0.7124041306441138, 80.}, {0.8173138546173232, 80.}, {
        0.9151727315943586, 80.}, {1.0212868840241998`, 80.}, {
        1.120350189457867, 80.}, {1.2174646887116864`, 80.}, {
        1.3228344634183116`, 80.}, {1.4211533911287626`, 80.}, {
        1.5277275942920192`, 80.}, {1.6323529912754284`, 80.}, {
        1.7299275412626636`, 80.}, {1.8357573667027043`, 80.}, {
        1.934536345146571, 80.}, {2.041570599043243, 80.}, {2.146656046760068,
         80.}, {2.2446906474807187`, 80.}, {2.350980523654175, 80.}, {
        2.450219552831457, 80.}, {2.547509775828892, 80.}, {
        2.6530552742791325`, 80.}, {2.751549925733199, 80.}, {
        2.8582998526400707`, 80.}, {2.963100973367095, 80.}, {
        3.0608512470979456`, 80.}, {3.1668567962816017`, 80.}, {
        3.265811498469084, 80.}, {3.3628173944767186`, 80.}, {
        3.4680785659371587`, 80.}, {3.566288890401425, 80.}, {
        3.672754490318497, 80.}, {3.772169243239395, 80.}, {
        3.8696351899804458`, 80.}, {3.975356412174302, 80.}, {
        4.074026787371984, 80.}, {4.180952438022472, 80.}, {4.285929282493112,
         80.}, {4.383855279967578, 80.}, {4.490036552894849, 80.}, {
        4.589166978825947, 80.}, {4.686348598577197, 80.}, {4.791785493781253,
         80.}, {4.890171541989135, 80.}, {4.996812865649822, 80.}, {
        4.998406381804503, 80.}, {4.999999897959183, 80.}}]}},
     AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
     Axes->True,
     AxesOrigin->{0, 0},
     DisplayFunction:>$DisplayFunction,
     FormatType->TraditionalForm,
     PlotRange->{{0, 5}, {0., 999.9999591836738}},
     PlotRangeClipping->True,
     PlotRangePadding->{
       Scaled[0.02], 
       Scaled[0.02]}], {-1, -0.6180339887498948}, {
    Left, Bottom}, {2, 1.2360679774997896`}], {
    {GrayLevel[0], 
     RectangleBox[{1.04, -0.35}, {1.770434782608696, 0.4500000000000001}]}, 
    {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.001]}], 
     RectangleBox[{0.99, -0.3}, {1.7204347826086956`, 0.5000000000000002}], 
     InsetBox[
      GraphicsBox[{{{InsetBox[
           GraphicsBox[
            {Hue[0.1421359549995791, 0.6, 0.6], Thickness[0.002], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 0.08}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"f''\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 0.58}, {-1, 0}, Automatic, {1, 0}]}, {
          InsetBox[
           GraphicsBox[
            {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.002], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 1.24}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"f'\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 1.74}, {-1, 0}, Automatic, {1, 0}]}, {
          InsetBox[
           GraphicsBox[
            {Hue[0.67, 0.6, 0.6], Thickness[0.005], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 2.4}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"f\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 2.9}, {-1, 0}, Automatic, {1, 0}]}}, {}},
       AspectRatio->1.0952380952380953`,
       FormatType->TraditionalForm,
       PlotRange->{{-0.1, 3.26}, {-0.1, 
        3.5799999999999996`}}], {0.99, -0.3}, {
      Left, Bottom}, {0.7304347826086958, 0.8000000000000002}]}}},
  AspectRatio->Automatic,
  Background->None,
  ColorOutput->Automatic,
  ImageSize->Automatic,
  PlotRange->All]], "Output",
 CellChangeTimes->{{3.41548122972036*^9, 3.4154812502298512`*^9}, 
   3.4154813435840883`*^9, {3.41555597633888*^9, 3.4155559989013233`*^9}, 
   3.415556273836661*^9},
 Magnification->1.5]
}, Open  ]],

Cell["\<\

Let' s try a more complicated function to see how f' and f'' behave
\
\>", "Input",
 CellChangeTimes->{{3.415480499570456*^9, 3.4154805033659134`*^9}, {
  3.4154805389270477`*^9, 3.4154805593063517`*^9}},
 FormatType->"TextForm",
 Magnification->1.5],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{"Concave", " ", "or", " ", "concave", " ", "down"}], " ", "*)"}], 
  "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
    RowBox[{
    "Notice", " ", "where", " ", "the", " ", "velocity", " ", "is", " ", 
     "zero"}], ",", " ", 
    RowBox[{
    "and", " ", "that", " ", "the", " ", "acceleration", " ", "is", " ", 
     "always", " ", "negative", " ", "for", " ", "the", " ", "concave", " ", 
     "function"}]}], " ", "*)"}], "\[IndentingNewLine]", 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"g", "[", "t_", "]"}], " ", ":=", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"t", "^", "2"}]}], " ", "+", 
     RowBox[{"2", "t"}], " ", "+", "1"}]}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"gderiv", "[", "t_", "]"}], " ", ":=", " ", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "2"}], "t"}], " ", "+", " ", "2"}]}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"gsecderiv", "[", "t_", "]"}], " ", ":=", " ", 
    RowBox[{"-", "2"}]}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"g", "[", "t", "]"}], ",", " ", 
       RowBox[{"gderiv", "[", "t", "]"}], ",", " ", 
       RowBox[{"gsecderiv", "[", "t", "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", 
       RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", " ", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Thickness", "[", "0.005", "]"}], ",", " ", 
        RowBox[{"Thickness", "[", "0.002", "]"}], ",", 
        RowBox[{"Thickness", "[", "0.002", "]"}]}], "}"}]}], ",", " ", 
     RowBox[{"PlotLegend", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Style", "[", 
         RowBox[{"\"\<g\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}], 
        ",", " ", 
        RowBox[{"Style", "[", 
         RowBox[{"\"\<g'\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}], 
        ",", " ", 
        RowBox[{"Style", "[", 
         RowBox[{"\"\<g''\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}]}], 
       "}"}]}], ",", 
     RowBox[{"LegendPosition", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"0.99", ",", " ", 
        RowBox[{"-", "0.3"}]}], "}"}]}]}], " ", "]"}], 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.4154805644237103`*^9, 3.415480649736384*^9}, {
   3.415480788075306*^9, 3.4154808368654623`*^9}, 3.4154813322578015`*^9, {
   3.4154813714541636`*^9, 3.4154813811080446`*^9}, {3.415481517514187*^9, 
   3.4154815176143312`*^9}, {3.4154816377570877`*^9, 3.415481652528328*^9}, {
   3.4154817670930643`*^9, 3.4154817788800125`*^9}},
 Magnification->1.5],

Cell[BoxData[
 GraphicsBox[{InsetBox[
    GraphicsBox[{{}, {}, 
      {Hue[0.67, 0.6, 0.6], Thickness[0.005], LineBox[CompressedData["
1:eJwV1nk8VG0bB3DLIEu24XmkRSHblCRbUtfIXpJCi0RJEqJIEfGUJWQp+5Il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        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.002], 
       LineBox[CompressedData["
1:eJwt1Ps31Hkcx3EmMphc5ouQfDHWl3HEKY1N5fsWDnYl67Zn2eiQ2mpcTrXS
Comi63HdcU3KzilSu3VI1vb9uKQ2iz2LURubS0YXCbnfZj97vvvD6zzP4x94
mUfE+kdxlJSUfPH+a9Lo7imFgkBsKZAnGsy8XCYQWwp8dV7NPVggEFsKzLbF
rcR+IhBbCpou5HAH5ARiSwF3U68paicQWwryju/78nQxgdhSUKV8rELJiUBs
KVCN8txstJeP2FoB8fyx+Hq2LmL7Gfg3F0XaPNNBbC0h8eNOcdawNno0Zyp6
OimAuhNFh+/u0UJ923fX9FIC8M0JMlst5CHqd6GOwscCPEt/8/Ua00AVHWGU
Zbw5SJ+clB0LVUddwzveP6w2g7D0PP3Om2qILQnJ+9bc0STWIramoGRHMm5n
VRBbE/j7YmFGVQ8HsTUGE43vbk3vUkZsjeDIVqHqSxMFU2ucfyb6ynqIdPrn
ZnPzEnNvQ2z34KA+CBOcFTEV84z+jtIUJ389KO8PaDnpOsP8OW4z1HCND6Op
+js8DaeYaaRxVcDThfp8a483wR+Y/i/yLI5XaYHD2xfT9qVypmBavCjx1QT3
+pfqK8EDjEX/h/RkWy7E3bo7r0t0Md++s01uaVGB4AOC8r9OPWSiRHxhu0QZ
nvzx4sKWgDr6YGvFlcKZZdrZrvGen6SLjjxUJv3acZbmxOv37TcaoJeJHk9R
+0dajeeXGjEup6dGw1ecBUN02oDM7eiND/SbJa/B6NdN9M/F5/ySpyZppV8P
cUeoTib2+n6d18EzdMP5HNes0BEm7HDRxozUebqh23G71ppJ5vxX8du2lCzR
/BtjRw98Pseo6OXLUxZXaa61sF0xvcL0tp2dSNdWhjsSzUF1koOCEiNERc0c
qJxtETtmqCKx5nuLjhAVeG4g8yip4SKyqvXHte9UwSX4st26bk2UWOcWoF2s
BrNxGVJTkTaKk4prRfbqwFYX8ThOEZtGNIAtH3lEB+YLc3nAVg8ZPtwY62Gp
BWwNUM7RzGyHXm1gux75n9tjV1qrAxW8shDnaCMkDE0PqkjShcBabpXXkjGS
ydWv5wbx4XY74U5VmaCOiBFXPklAyeNnaZYupqgoX3ftyQUCHG0FvKkpEg2d
azLPxW5Zd9bN6hOJbBKO7azG9p+Q/xCCXRfSc3wQO66m8m0TtowsHPZeJOA2
vflJ7jSJiEqzRuMlAqwCXc9snSXRpUf2pxqWCTBMCp9PWCBR990BiQz7Znij
fTX2hvLs+xPYol2Cg4PYlWmf3lmuEBCoNtrjvUiip14PvrmMnZUdc994iUQq
XS5OYasEcKVJMQ3LJPJp+eifgF2Q+eqnCezcmmsxOdhWR1z7LVdIZFHAkbZi
uzuo+lzGhr2tevYKAk7XX7IJWyVRpu8JB29sndLxfTnYnbS1TyT21RS/glbs
MIsLaRLsBnc9NXsFiaTE9rJfsH2oeJdI7HGVsfo27D71599LsEWzJbIRbPHY
tuo27P//DBY7il8rsP8F5sRUAg==
        "]]}, 
      {Hue[0.14213595499957954`, 0.6, 0.6], Thickness[0.002], 
       LineBox[{{-4.9999997959183675`, -2.}, {-4.901856591951532, -2.}, \
{-4.803713387984696, -2.}, {-4.590916429145413, -2.}, {-4.392221164298479, \
-2.}, {-4.197423511811239, -2.}, {-3.986115308418389, -2.}, \
{-3.788908799017886, -2.}, {-3.575191738711772, -2.}, {-3.3653722907653534`, \
-2.}, {-3.169654536811282, -2.}, {-2.9574262319516, -2.}, \
{-2.759299621084266, -2.}, {-2.565070622576627, -2.}, {-2.354331073163377, \
-2.}, {-2.157693217742475, -2.}, {-1.9445448114159614`, -2.}, \
{-1.735294017449143, -2.}, {-1.5401449174746724`, -2.}, \
{-1.3284852665945908`, -2.}, {-1.130927309706857, -2.}, {-0.9168588019135122, \
-2.}, {-0.7066879064798623, -2.}, {-0.5106187050385604, -2.}, \
{-0.2980389526916474, -2.}, {-0.09956089433708235, -2.}, {
        0.09501955165778773, -2.}, {0.306110548558269, -2.}, {
        0.5030998514664022, -2.}, {0.7165997052801466, -2.}, {
        0.9262019467341961, -2.}, {1.1217024941958975`, -2.}, {
        1.3337135925632102`, -2.}, {1.5316229969381747`, -2.}, {
        1.7256347889534445`, -2.}, {1.9361571318743254`, -2.}, {
        2.132577780802858, -2.}, {2.345508980637002, -2.}, {
        2.5443384864787983`, -2.}, {2.739270379960899, -2.}, {
        2.9507128243486114`, -2.}, {3.148053574743976, -2.}, {
        3.3619048760449513`, -2.}, {3.5718585649862313`, -2.}, {
        3.767710559935164, -2.}, {3.9800731057897076`, -2.}, {
        4.178333957651903, -2.}, {4.3726971971544035`, -2.}, {
        4.583570987562515, -2.}, {4.780343083978279, -2.}, {
        4.9936257312996535`, -2.}, {4.996812763609011, -2.}, {
        4.9999997959183675`, -2.}}]}},
     AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
     Axes->True,
     AxesOrigin->{0, 0},
     DisplayFunction:>$DisplayFunction,
     FormatType->TraditionalForm,
     PlotRange->{{-5, 5}, {-18.083456256859204`, 11.999999591836735`}},
     PlotRangeClipping->True,
     PlotRangePadding->{
       Scaled[0.02], 
       Scaled[0.02]}], {-1, -0.6180339887498948}, {
    Left, Bottom}, {2, 1.2360679774997896`}], {
    {GrayLevel[0], 
     RectangleBox[{1.04, -0.35}, {1.770434782608696, 0.4500000000000001}]}, 
    {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.001]}], 
     RectangleBox[{0.99, -0.3}, {1.7204347826086956`, 0.5000000000000002}], 
     InsetBox[
      GraphicsBox[{{{InsetBox[
           GraphicsBox[
            {Hue[0.1421359549995791, 0.6, 0.6], Thickness[0.002], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 0.08}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"g''\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 0.58}, {-1, 0}, Automatic, {1, 0}]}, {
          InsetBox[
           GraphicsBox[
            {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.002], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 1.24}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"g'\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 1.74}, {-1, 0}, Automatic, {1, 0}]}, {
          InsetBox[
           GraphicsBox[
            {Hue[0.67, 0.6, 0.6], Thickness[0.005], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 2.4}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"g\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 2.9}, {-1, 0}, Automatic, {1, 0}]}}, {}},
       AspectRatio->1.0952380952380953`,
       FormatType->TraditionalForm,
       PlotRange->{{-0.1, 3.26}, {-0.1, 
        3.5799999999999996`}}], {0.99, -0.3}, {
      Left, Bottom}, {0.7304347826086958, 0.8000000000000002}]}}},
  AspectRatio->Automatic,
  Background->None,
  ColorOutput->Automatic,
  ImageSize->Automatic,
  PlotRange->All]], "Output",
 CellChangeTimes->{
  3.41548066031159*^9, {3.4154808231357203`*^9, 3.415480837396226*^9}, 
   3.415481512697261*^9, 3.415481564782155*^9, 3.415556007754053*^9, 
   3.415556279104235*^9},
 Magnification->1.5]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{"Convex", " ", "or", " ", "Concave", " ", "up"}], " ", "*)"}], " ",
   "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
    RowBox[{
    "Notice", " ", "where", " ", "the", " ", "velocity", " ", "is", " ", 
     "zero"}], ",", " ", 
    RowBox[{
    "and", " ", "that", " ", "the", " ", "acceleration", " ", "is", " ", 
     "always", " ", "positive", " ", "for", " ", "this", " ", "convex", " ", 
     "function"}]}], " ", "*)"}], "\[IndentingNewLine]", 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"g2", "[", "t_", "]"}], " ", ":=", 
    RowBox[{
     RowBox[{"t", "^", "2"}], " ", "+", 
     RowBox[{"2", "t"}], " ", "+", "1"}]}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"g2deriv", "[", "t_", "]"}], " ", ":=", " ", 
    RowBox[{
     RowBox[{"2", "t"}], " ", "+", " ", "2"}]}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"g2secderiv", "[", "t_", "]"}], " ", ":=", " ", "2"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"g2", "[", "t", "]"}], ",", " ", 
       RowBox[{"g2deriv", "[", "t", "]"}], ",", " ", 
       RowBox[{"g2secderiv", "[", "t", "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", 
       RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", " ", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Thickness", "[", "0.005", "]"}], ",", " ", 
        RowBox[{"Thickness", "[", "0.002", "]"}], ",", 
        RowBox[{"Thickness", "[", "0.002", "]"}]}], "}"}]}], ",", " ", 
     RowBox[{"PlotLegend", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Style", "[", 
         RowBox[{"\"\<g\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}], 
        ",", " ", 
        RowBox[{"Style", "[", 
         RowBox[{"\"\<g'\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}], 
        ",", " ", 
        RowBox[{"Style", "[", 
         RowBox[{"\"\<g''\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}]}], 
       "}"}]}], ",", 
     RowBox[{"LegendPosition", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"0.99", ",", " ", 
        RowBox[{"-", "0.3"}]}], "}"}]}]}], " ", "]"}], 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.4154815922616687`*^9, 3.4154816313578863`*^9}, {
  3.4154816644554787`*^9, 3.4154817024701405`*^9}, {3.415481733174291*^9, 
  3.4154817902062993`*^9}},
 Magnification->1.5],

Cell[BoxData[
 GraphicsBox[{InsetBox[
    GraphicsBox[{{}, {}, 
      {Hue[0.67, 0.6, 0.6], Thickness[0.005], LineBox[CompressedData["
1:eJwV1ndYjW0cB/DTkLSHUtGSUGkgLfF7UopSWtIrkhQtOdXbUOEgIfGqlBaO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        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.002], 
       LineBox[CompressedData["
1:eJwt1Pkz3Hccx/G11rWM3fV1RJB1xReJEMaVVr7vRIyViGMdnZKYjKPTUcVg
HCM0xpWSmLoirhY1RmhC09adfD9xdhJEK1lRSbNNYtWRECJu22/n+/3hNY95
/gMvk/A4cRSbxWL5UPvfjLnzq3I5hrbHq9/I5UIkS9ddn9nF0OjG0YXOLSHy
4b/c6NzCUI3x/eW4NSEydo3fi1vDECRIt1/KhKi/oERVKsNQjs5hARoTItVj
U4fQGIb4F+64XakWorKkS+euVGPIch7dYDkLUatCYiPLGUPBijKRQuwhpBTl
aa9/UQtxHxZF2vcYIuzZUExDsQBZtBdOnzEzQOKBqgirh3zUmOc4Ii7TR+nL
bjHfveYx6qGulKroNl9NRl3kUxJkvF+pwaiNPGvv+YiWuIxaqGkkTZIYqsYo
QGE5ZTqPm1VQn6rVke/9eSjzkuIddUwZ7SLlC3oWGohlIyTdczmIFXIyM5+r
hv4qrMxvfcpGv+o58D5rVUKG3C9vfTitgFy0J8TzuWz0laO10oyhnJREB3I7
XPfJCOe/mwcGdsjh2D7//PwN0jr1hDy2cZMMXEnafyF6T9a/CBhMO7VO5iT7
CuaUZsm5LJ1PPQ+ski68n6Su/RNkT7mlx7/Bb0k7ca/knmiIsJuf/mBbKyPr
bLx164ZeEWd6ZtT2gqVkaqFoSUm8TMTfatsUYJOk6EizYkryRyL4C7P6Py93
k46lM61RE7vEyOh0gUNAFzFac9vFIVcBTtg8uOtXMUn4nGs/+mSWA+xkneeR
+lJCoZcbe1dDFVQ0/LLC38kIjnIzHE9Rh2ypxD3hx7fE6fhFk+4/NKG9Os8v
c/U98ey8uyWxxYe4hkj+m+B1wsXfXjWhVQvCoquM8rM2CVvHsaIB0IZv/ZNd
HWp2iEHL3hr9cR3gaJfLvtneJ9gFZ9eIDD2YepS7ksNTYNSHoPRwp6oBNuNB
iFFfNB0P4TAagrB1+IbyghKjEaR3uQfwqlUYhRDfFNPhZKsGLgFOodUVxqDB
dg4/NssF5SyD4xUJJuDxdWC5dakGNDi1nC33MoUD3UZxHuaacO15UaeplhmU
JFwttpvigZfroOjmohmI83xtajv4jOZgHZoT1JghYDwMEplaQ2mQFqMFjIfP
ntISYow4VJULlNO2MEYcXuX1m5RSTYuDVWqi222qaXHoCnma9A/VtDhIhJWv
vbYxRhywFuMHB3cwRhyu3be93LeLMeLwpE1aIaGaFgeD+uJfVqimxaEle23B
fA9jxOF3Uefn16mmxYEzedI5bB9jxMF7cFmcSjUtDqW/1cWWUE2Lg+lNdtMw
1bQ4wMVhbVs5xojDVZ8UOy+qaXF4TFh6R1BNi0OYaUF2BdW0ODRhn/zwM9W0
OLzjLPU8opoWB6ePNZJZqmlxYP6MEYf/ANJRSr4=
        "]]}, 
      {Hue[0.14213595499957954`, 0.6, 0.6], Thickness[0.002], 
       LineBox[{{-4.9999997959183675`, 2.}, {-4.901856591951532, 
        2.}, {-4.803713387984696, 2.}, {-4.590916429145413, 
        2.}, {-4.392221164298479, 2.}, {-4.197423511811239, 
        2.}, {-3.986115308418389, 2.}, {-3.788908799017886, 
        2.}, {-3.575191738711772, 2.}, {-3.3653722907653534`, 
        2.}, {-3.169654536811282, 2.}, {-2.9574262319516, 
        2.}, {-2.759299621084266, 2.}, {-2.565070622576627, 
        2.}, {-2.354331073163377, 2.}, {-2.157693217742475, 
        2.}, {-1.9445448114159614`, 2.}, {-1.735294017449143, 
        2.}, {-1.5401449174746724`, 2.}, {-1.3284852665945908`, 
        2.}, {-1.130927309706857, 2.}, {-0.9168588019135122, 
        2.}, {-0.7066879064798623, 2.}, {-0.5106187050385604, 
        2.}, {-0.2980389526916474, 2.}, {-0.09956089433708235, 2.}, {
        0.09501955165778773, 2.}, {0.306110548558269, 2.}, {
        0.5030998514664022, 2.}, {0.7165997052801466, 2.}, {
        0.9262019467341961, 2.}, {1.1217024941958975`, 2.}, {
        1.3337135925632102`, 2.}, {1.5316229969381747`, 2.}, {
        1.7256347889534445`, 2.}, {1.9361571318743254`, 2.}, {
        2.132577780802858, 2.}, {2.345508980637002, 2.}, {2.5443384864787983`,
         2.}, {2.739270379960899, 2.}, {2.9507128243486114`, 2.}, {
        3.148053574743976, 2.}, {3.3619048760449513`, 2.}, {
        3.5718585649862313`, 2.}, {3.767710559935164, 2.}, {
        3.9800731057897076`, 2.}, {4.178333957651903, 2.}, {
        4.3726971971544035`, 2.}, {4.583570987562515, 2.}, {4.780343083978279,
         2.}, {4.9936257312996535`, 2.}, {4.996812763609011, 2.}, {
        4.9999997959183675`, 2.}}]}},
     AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
     Axes->True,
     AxesOrigin->{0, 0},
     DisplayFunction:>$DisplayFunction,
     FormatType->TraditionalForm,
     PlotRange->{{-5, 5}, {-7.999999591836735, 21.99847273697083}},
     PlotRangeClipping->True,
     PlotRangePadding->{
       Scaled[0.02], 
       Scaled[0.02]}], {-1, -0.6180339887498948}, {
    Left, Bottom}, {2, 1.2360679774997896`}], {
    {GrayLevel[0], 
     RectangleBox[{1.04, -0.35}, {1.770434782608696, 0.4500000000000001}]}, 
    {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.001]}], 
     RectangleBox[{0.99, -0.3}, {1.7204347826086956`, 0.5000000000000002}], 
     InsetBox[
      GraphicsBox[{{{InsetBox[
           GraphicsBox[
            {Hue[0.1421359549995791, 0.6, 0.6], Thickness[0.002], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 0.08}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"g''\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 0.58}, {-1, 0}, Automatic, {1, 0}]}, {
          InsetBox[
           GraphicsBox[
            {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.002], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 1.24}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"g'\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 1.74}, {-1, 0}, Automatic, {1, 0}]}, {
          InsetBox[
           GraphicsBox[
            {Hue[0.67, 0.6, 0.6], Thickness[0.005], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 2.4}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"g\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 2.9}, {-1, 0}, Automatic, {1, 0}]}}, {}},
       AspectRatio->1.0952380952380953`,
       FormatType->TraditionalForm,
       PlotRange->{{-0.1, 3.26}, {-0.1, 
        3.5799999999999996`}}], {0.99, -0.3}, {
      Left, Bottom}, {0.7304347826086958, 0.8000000000000002}]}}},
  AspectRatio->Automatic,
  Background->None,
  ColorOutput->Automatic,
  ImageSize->Automatic,
  PlotRange->All]], "Output",
 CellChangeTimes->{{3.4154816810893965`*^9, 3.4154817040824594`*^9}, 
   3.415556016496624*^9, 3.415556284762371*^9},
 Magnification->1.5]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{
   "This", " ", "example", " ", "is", " ", "more", " ", "complicated"}], " ", 
   "*)"}], "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Notice", " ", "the", " ", "local", " ", "max", " ", "and", " ", "min", 
    " ", "function", " ", "values", " ", "and", " ", "the", " ", 
    "corresponding", " ", "first", " ", "derivative", " ", "there"}], " ", 
   "*)"}], "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Acceleration", " ", "is", " ", "zero", " ", "where", " ", "the", " ", 
    "function", " ", "changes", " ", "from", " ", "concave", " ", "to", " ", 
    "convex", " ", "or", " ", "visa", " ", 
    RowBox[{"versa", "."}]}], " ", "*)"}], "\[IndentingNewLine]", 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"hfct", "[", "t_", "]"}], " ", ":=", 
    RowBox[{
     RowBox[{"3", 
      RowBox[{"t", "^", "4"}]}], " ", "-", " ", 
     RowBox[{"4", 
      RowBox[{"t", "^", "3"}]}], " ", "-", 
     RowBox[{"12", 
      RowBox[{"t", "^", "2"}]}], " ", "+", " ", "8"}]}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"hderiv", "[", "t_", "]"}], " ", ":=", " ", 
    RowBox[{
     RowBox[{"12", 
      RowBox[{"t", "^", "3"}]}], " ", "-", 
     RowBox[{"12", 
      RowBox[{"t", "^", "2"}]}], " ", "-", 
     RowBox[{"24", "t"}]}]}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"hsecderiv", "[", "t_", "]"}], " ", ":=", " ", 
    RowBox[{
     RowBox[{"36", 
      RowBox[{"t", "^", "2"}]}], " ", "-", 
     RowBox[{"24", "t"}], " ", "-", "24"}]}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"hfct", "[", "t", "]"}], ",", " ", 
       RowBox[{
        RowBox[{"h", "'"}], "[", "t", "]"}], ",", " ", 
       RowBox[{"hsecderiv", "[", "t", "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", 
       RowBox[{"-", "3"}], ",", "3"}], "}"}], ",", " ", "\[IndentingNewLine]",
      
     RowBox[{"PlotRange", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "30"}], ",", "30"}], "}"}]}], ",", "\[IndentingNewLine]",
      
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Thickness", "[", "0.005", "]"}], ",", " ", 
        RowBox[{"Thickness", "[", "0.002", "]"}], ",", 
        RowBox[{"Thickness", "[", "0.002", "]"}]}], "}"}]}], ",", " ", 
     RowBox[{"PlotLegend", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Style", "[", 
         RowBox[{"\"\<h\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}], 
        ",", " ", 
        RowBox[{"Style", "[", 
         RowBox[{"\"\<h'\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}], 
        ",", " ", 
        RowBox[{"Style", "[", 
         RowBox[{"\"\<h''\>\"", ",", " ", "15", ",", "  ", 
          RowBox[{"FontFamily", "\[Rule]", "\"\<Helvetica\>\""}]}], "]"}]}], 
       "}"}]}], ",", 
     RowBox[{"LegendPosition", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"0.99", ",", " ", 
        RowBox[{"-", "0.3"}]}], "}"}]}]}], " ", "]"}], 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.415480847851259*^9, 3.415480874789995*^9}, {
  3.4154809173211517`*^9, 3.415481039967509*^9}, {3.4154811597697763`*^9, 
  3.415481221278221*^9}, {3.4154812730726976`*^9, 3.415481308223242*^9}, {
  3.4154813874571743`*^9, 3.4154813933556557`*^9}, {3.4154818041563587`*^9, 
  3.4154818476689262`*^9}, {3.4154820056661153`*^9, 3.415482070960003*^9}, {
  3.4154821180276833`*^9, 3.415482132889053*^9}, {3.4154822329829807`*^9, 
  3.4154823503717775`*^9}, {3.4155559442527423`*^9, 3.415555946626155*^9}, {
  3.4155560436256337`*^9, 3.4155561037921486`*^9}, {3.415556402471629*^9, 
  3.415556408059664*^9}, {3.415558802658675*^9, 3.4155588060735855`*^9}},
 Magnification->1.5],

Cell[BoxData[
 GraphicsBox[{InsetBox[
    GraphicsBox[{{}, {}, 
      {Hue[0.67, 0.6, 0.6], Thickness[0.005], LineBox[CompressedData["
1:eJw1mnk01F/cx8c+1iTZGUpIhWzTgvspilJIKkWIFhVRP1EikigJlYQkJBQh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        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.002], 
       LineBox[CompressedData["
1:eJwtmXk4VF/8x1GWkLTZZxLZxr6MmUGdky0hoU2iKIUQKfsSklZCJZWvUAmp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        "]]}, 
      {Hue[0.14213595499957954`, 0.6, 0.6], Thickness[0.002], 
       LineBox[CompressedData["
1:eJw1l3k0lf0Wx2VKmiiEm6mkUobEcc55HFulIpLQ5JUQQoncCk0is1JUproZ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        "]]}},
     AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
     Axes->True,
     AxesOrigin->{0, 0},
     DisplayFunction:>$DisplayFunction,
     FormatType->TraditionalForm,
     PlotRange->{{-3, 3}, {-30, 30}},
     PlotRangeClipping->True,
     PlotRangePadding->{
       Scaled[0.02], Automatic}], {-1, -0.6180339887498948}, {
    Left, Bottom}, {2, 1.2360679774997896`}], {
    {GrayLevel[0], 
     RectangleBox[{1.04, -0.35}, {1.770434782608696, 0.4500000000000001}]}, 
    {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.001]}], 
     RectangleBox[{0.99, -0.3}, {1.7204347826086956`, 0.5000000000000002}], 
     InsetBox[
      GraphicsBox[{{{InsetBox[
           GraphicsBox[
            {Hue[0.1421359549995791, 0.6, 0.6], Thickness[0.002], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 0.08}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"h''\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 0.58}, {-1, 0}, Automatic, {1, 0}]}, {
          InsetBox[
           GraphicsBox[
            {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.002], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 1.24}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"h'\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 1.74}, {-1, 0}, Automatic, {1, 0}]}, {
          InsetBox[
           GraphicsBox[
            {Hue[0.67, 0.6, 0.6], Thickness[0.005], 
             LineBox[{{0, 0}, {1, 0}}]}], {0.08, 2.4}, {
           Left, Bottom}, {1, 1}], InsetBox[
           StyleBox["\<\"h\"\>",
            StripOnInput->False,
            FontFamily->"Helvetica",
            FontSize->15], {1.21, 2.9}, {-1, 0}, Automatic, {1, 0}]}}, {}},
       AspectRatio->1.0952380952380953`,
       FormatType->TraditionalForm,
       PlotRange->{{-0.1, 3.26}, {-0.1, 
        3.5799999999999996`}}], {0.99, -0.3}, {
      Left, Bottom}, {0.7304347826086958, 0.8000000000000002}]}}},
  AspectRatio->Automatic,
  Background->None,
  ColorOutput->Automatic,
  ImageSize->Automatic,
  PlotRange->All]], "Output",
 CellChangeTimes->{{3.4154820416979265`*^9, 3.415482073303373*^9}, {
   3.415482121632867*^9, 3.4154821341007953`*^9}, 3.4155559517334995`*^9, {
   3.415556025169094*^9, 3.415556105624784*^9}, 3.4155562993433375`*^9, 
   3.4155564103229184`*^9},
 Magnification->1.5]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"k", "[", "x_", "]"}], " ", ":=", " ", 
     RowBox[{
      RowBox[{"3", 
       RowBox[{"x", "^", "2"}]}], " ", "+", " ", 
      RowBox[{"2", 
       RowBox[{"Sin", "[", 
        RowBox[{"5", "x"}], "]"}]}]}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"Manipulate", "[", 
    RowBox[{
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"k", "[", "x", "]"}], ",", " ", 
         RowBox[{
          RowBox[{"k", "'"}], "[", "x", "]"}]}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "b"}], "}"}]}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"b", ",", "1"}], "}"}], ",", "1", ",", "10"}], "}"}]}], 
    "]"}]}]}]], "Input",
 CellChangeTimes->{{3.41555277023197*^9, 3.4155528785577345`*^9}}],

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`b$$ = 1., Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`b$$], 1}, 1, 10}}, Typeset`size$$ = {
    450., {133., 140.}}, Typeset`update$$ = 0, Typeset`initDone$$, 
    Typeset`skipInitDone$$ = True, $CellContext`b$4807$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, "Variables" :> {$CellContext`b$$ = 1}, 
      "ControllerVariables" :> {
        Hold[$CellContext`b$$, $CellContext`b$4807$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> Plot[{
         $CellContext`k[$CellContext`x], 
         Derivative[1][$CellContext`k][$CellContext`x]}, {$CellContext`x, 
         0, $CellContext`b$$}], 
      "Specifications" :> {{{$CellContext`b$$, 1}, 1, 10}}, "Options" :> {}, 
      "DefaultOptions" :> {}],
     ImageSizeCache->{499., {183.5, 190.5}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellChangeTimes->{
  3.41555285860905*^9, 3.4155528906751585`*^9, 3.415556816747328*^9, {
   3.4155568580767565`*^9, 3.415556900367568*^9}, {3.4155570083828864`*^9, 
   3.4155570125889344`*^9}, {3.4155570594062543`*^9, 3.4155570738770623`*^9}, 
   3.427041934288706*^9}]
}, Open  ]]
},
WindowSize->{863, 700},
WindowMargins->{{Automatic, 106}, {Automatic, 95}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 1891, 29, 582, "Input"],
Cell[CellGroupData[{
Cell[2484, 54, 2131, 53, 383, "Input"],
Cell[4618, 109, 9366, 173, 279, "Output"]
}, Open  ]],
Cell[13999, 285, 261, 8, 99, "Input"],
Cell[CellGroupData[{
Cell[14285, 297, 2883, 72, 487, "Input"],
Cell[17171, 371, 11304, 206, 275, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[28512, 582, 2649, 67, 487, "Input"],
Cell[31164, 651, 10816, 197, 275, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[42017, 853, 3943, 97, 545, "Input"],
Cell[45963, 952, 27678, 477, 275, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[73678, 1434, 936, 29, 115, "Input"],
Cell[74617, 1465, 1857, 38, 408, "Output"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
