(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     43229,        903]
NotebookOptionsPosition[     42580,        879]
NotebookOutlinePosition[     42998,        895]
CellTagsIndexPosition[     42955,        892]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell["\<\


Mathematical Principles for Scientific Computing and Visualization  
http://www.farinhansford.com/books/scv
by Gerald Farin and Dianne Hansford

This notebook demonstrates finding roots with Newton-Raphson method.
Given is a function f and a guess x-value of where the root might live.
This method attempts to find where f intersects the line y=0 within a given \
number of iterations.

In the first cell, we have an  example where Newton's method is successful in \
finding the root. In the second cell, we have an example where Newton's \
method fails.

Figures 10.5 and 10.6 were generated with this notebook.

See Chapter 10: Finding Roots for details.

Updated July 2008\
\>", "Input",
 CellChangeTimes->{{3.4263420672253327`*^9, 3.4263420926518946`*^9}, {
  3.4263423818176947`*^9, 3.4263423823885155`*^9}, {3.4263427876111965`*^9, 
  3.42634281333819*^9}, {3.426344780436739*^9, 3.426344821185333*^9}, {
  3.426345059227621*^9, 3.4263451229192047`*^9}, {3.426345264933411*^9, 
  3.4263453026676702`*^9}},
 FormatType->"TextForm"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{"Specify", " ", "the", " ", "output", " ", "folder"}], " ", "*)"}],
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"SetDirectory", "[", 
     RowBox[{"ToFileName", "[", 
      RowBox[{"Extract", "[", 
       RowBox[{
        RowBox[{"\"\<FileName\>\"", "/.", "\[InvisibleSpace]", 
         RowBox[{"NotebookInformation", "[", 
          RowBox[{"EvaluationNotebook", "[", "]"}], "]"}]}], ",", 
        RowBox[{"{", "1", "}"}], ",", "FrontEnd`FileName"}], "]"}], "]"}], 
     "]"}], ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Create", " ", "a", " ", 
     RowBox[{"function", " ", "--"}], " ", "we", " ", "want", " ", "the", " ",
      "roots", " ", "of", " ", "this"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"f", "[", "x_", "]"}], ":=", 
     RowBox[{"N", "[", 
      RowBox[{
       SuperscriptBox["x", "2"], "-", "0.5"}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Set", " ", "the", " ", "number", " ", "of", " ", "iterations"}], 
    " ", "*)"}], "\n", 
   RowBox[{
    RowBox[{"n", "=", "3"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Establish", " ", "the", " ", 
     RowBox[{"domain", " ", "[", 
      RowBox[{"a", ",", "b"}], "]"}], " ", "of", " ", "interest", " ", "for", 
     " ", "the", " ", "funciton", " ", "f"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"This", " ", "is", " ", "used", " ", "for", " ", "plotting"}], 
    " ", "*)"}], "\n", 
   RowBox[{
    RowBox[{"a", "=", "0"}], ";"}], "\n", 
   RowBox[{
    RowBox[{"b", "=", "2"}], ";"}], "\[IndentingNewLine]", "\n", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Create", " ", "a", " ", "guess", " ", "x"}], "-", 
     RowBox[{"value", " ", "for", " ", "the", " ", "root"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"xold", "=", "3.4"}], ";"}], "\[IndentingNewLine]", "\n", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "storage", " ", "for", " ", "the", " ", "iterations"}], 
    " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"points", "=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0.0", ",", "0.0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", 
         RowBox[{"2", " ", "n"}]}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Newton", "-", 
     RowBox[{"Raphson", " ", 
      RowBox[{"method", " ", "--"}], " ", "see", " ", "Section", " ", 
      "10.2"}]}], " ", "*)"}], "\n", 
   RowBox[{
    RowBox[{"Do", "[", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"xnew", "=", 
        RowBox[{"xold", "-", 
         FractionBox[
          RowBox[{"f", "[", "xold", "]"}], 
          RowBox[{
           SuperscriptBox["f", "\[Prime]",
            MultilineFunction->None], "[", "xold", "]"}]]}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"ynew", "=", 
        RowBox[{"f", "[", "xnew", "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"xold", "=", "xnew"}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"points", "\[LeftDoubleBracket]", 
         RowBox[{
          RowBox[{
           RowBox[{"2", " ", "i"}], "-", "1"}], ",", "1"}], 
         "\[RightDoubleBracket]"}], "=", "xnew"}], ";", "\[IndentingNewLine]",
        
       RowBox[{
        RowBox[{"points", "\[LeftDoubleBracket]", 
         RowBox[{
          RowBox[{
           RowBox[{"2", " ", "i"}], "-", "1"}], ",", "2"}], 
         "\[RightDoubleBracket]"}], "=", "0"}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"points", "\[LeftDoubleBracket]", 
         RowBox[{
          RowBox[{"2", " ", "i"}], ",", "1"}], "\[RightDoubleBracket]"}], "=",
         "xnew"}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"points", "\[LeftDoubleBracket]", 
         RowBox[{
          RowBox[{"2", " ", "i"}], ",", "2"}], "\[RightDoubleBracket]"}], "=",
         "ynew"}]}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
     "Plot", " ", "the", " ", "function", " ", "and", " ", "the", " ", "y"}], 
     "=", 
     RowBox[{"0", " ", "axis"}]}], " ", "*)"}], "\n", 
   RowBox[{
    RowBox[{"fctplot", "=", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"f", "[", "x", "]"}], ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "a", ",", "b"}], "}"}], ",", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.01", "]"}], ",", 
          RowBox[{"Thickness", "[", "0.001", "]"}]}], "}"}]}], ",", 
       RowBox[{"Axes", "\[Rule]", "False"}], ",", " ", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";"}], "\n", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Plot", " ", "the", " ", "points", " ", "joined", " ", "by", " ", "line", 
     " ", "segments"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"polyplot", "=", 
     RowBox[{"ListPlot", "[", 
      RowBox[{"points", ",", 
       RowBox[{"Joined", "\[Rule]", "True"}], ",", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"PointSize", "[", "0.03", "]"}], ",", " ", "Red"}], "}"}]}],
        ",", 
       RowBox[{"Axes", "\[Rule]", "False"}], ",", " ", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\n", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
     "Plot", " ", "the", " ", "points", " ", "generated", " ", "by", " ", 
      "the", " ", "N"}], "-", 
     RowBox[{"R", " ", "method"}]}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"pointplot", "=", 
     RowBox[{"ListPlot", "[", 
      RowBox[{"points", ",", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"PointSize", "[", "0.03", "]"}], ",", " ", "Red"}], "}"}]}],
        ",", 
       RowBox[{"Axes", "\[Rule]", "False"}], ",", " ", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\n", 
   RowBox[{"(*", " ", 
    RowBox[{"Put", " ", "the", " ", "plots", " ", "together"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"pic", "=", 
    RowBox[{"Show", "[", 
     RowBox[{"fctplot", ",", "pointplot", ",", "polyplot"}], "]"}]}], "\n", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Export", "[", 
     RowBox[{"\"\<Newton.eps\>\"", ",", "pic"}], "]"}], ";"}], 
   "\[IndentingNewLine]"}]}]], "Input",
 CellChangeTimes->{{3.4263447746283875`*^9, 3.426344776080475*^9}, 
   3.426344837438704*^9, {3.4263448763646765`*^9, 3.426344883895506*^9}, {
   3.4263449169630547`*^9, 3.4263450506352654`*^9}, {3.4263451371296387`*^9, 
   3.426345199889883*^9}, {3.4263452323765965`*^9, 3.42634524091888*^9}, {
   3.426345316647773*^9, 3.426345329576363*^9}, {3.4263501934502707`*^9, 
   3.426350194271451*^9}, {3.4263502295321536`*^9, 3.426350254578168*^9}, {
   3.4263507239731245`*^9, 3.426350735289397*^9}, {3.4263507672753906`*^9, 
   3.426350879006051*^9}, {3.426351217783189*^9, 3.4263512200264144`*^9}}],

Cell[BoxData[
 GraphicsBox[{{{}, {}, 
    {Hue[0.67, 0.6, 0.6], Thickness[0.01], LineBox[CompressedData["
1:eJwV1nk41ekXAHB7lztjK/WzJWtNDFKWMTi3oew1uIyki1QSERkjHiNKkSVm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      "]]}, 
    {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.001], 
     LineBox[CompressedData["
1:eJxN0GtI02EUBvC5NC1hOsPkP5kynUVpGCaals1SsLziJVJYM8xLKprJIA1E
J9pFXUSIjaZ5hTKdSzftg5mmy8hsalp22VytsZYTneW0edkqqPO+Bx4eft8e
DiPtYkIGmUQixf7J347M0E0Nfc84Rvp31mG0AhURzvrvQ42CegmRBI56PdvT
QKSDz6SvDdwmuGCl77UwPlEBbnKOu1pF1IJ5dpK9N4g2cHJKs9iwLAXbW3js
CLdBsFf8om6kWga+LA9iCVZegl2k1DkrqwlwnzpsfOLoG/BwToNrysIM2Lgn
xEGUNwsmmzocXb0+ggMs5/oSuxTg8Qp/+eh+Fdj9lt7UvvgZrF9Tzb09qAa3
zfePUW5+BVPo95c0Gg24Jrs3s4mpBa+GdmgulH4Dc/1k0t0jOrCqS1Y9xZgH
b+TMCPdd0YO9j9uaM/sXwBEtjKxul0X0L3qqxxf2Ethn4bT9bIwBvMvBYCdb
R7YeFIo965fBuf79H3ZE/QB3nEprerqKbNTTHLV1P8G9iRHbasNXwKUnKxul
WmSGMvJ9a6URHBxYUUv4rYLZbhMmq2lk4xhfXcBbAwusOdwE5i+0f663hSxD
tinhJQVdMoFJ7Ryx2mkdXHhXzHcaRh7wvm5Hzd8AUyMLfcopm2DFgcnp4ifI
j+6xmBrOFvisOIBKNyOXVTbzFA/NYJGOLqREW8DxVMrhF+vIVTZkVwVmmcm4
ZcAcoFaO0DaQaRJRXD5mdVJMlvMmMldQU3d+C1lUU1ZchFlbxmXzMSdnsz0e
Yw4J9unaaUbernj1vBsza3LwwSjmIpmk+hPm+U5hvI0FWV6Sq0zGbFuYOpSH
OTQzsbUcc0/skexOzPoTvtHPMDMDPX3fYeZ4uzjpMd9xtzdaMP8GDY1LRg==

      "]]}}, {{}, 
    {RGBColor[1, 0, 0], PointSize[0.03], 
     PointBox[{{1.773529411764706, 0.}, {1.773529411764706, 
      2.6454065743944644`}, {1.0277265632621209`, 0.}, {1.0277265632621209`, 
      0.55622188883457}, {0.7571186463717281, 0.}, {0.7571186463717281, 
      0.07322864468375789}}]}, {}}, {{}, {}, 
    {RGBColor[1, 0, 0], PointSize[0.03], 
     LineBox[{{1.773529411764706, 0.}, {1.773529411764706, 
      2.6454065743944644`}, {1.0277265632621209`, 0.}, {1.0277265632621209`, 
      0.55622188883457}, {0.7571186463717281, 0.}, {0.7571186463717281, 
      0.07322864468375789}}]}}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  AxesOrigin->{0, 0},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{
  3.4263502554594355`*^9, 3.4263507360204477`*^9, 3.426350768216744*^9, {
   3.4263508083544593`*^9, 3.4263508801777363`*^9}}]
}, Open  ]],

Cell["\<\

Newton's method fails in the next example.\
\>", "Input",
 CellChangeTimes->{{3.426344852189915*^9, 3.4263448700656195`*^9}},
 FormatType->"TextForm"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{"Create", " ", "a", " ", 
    RowBox[{"function", " ", "--"}], " ", "we", " ", "want", " ", "the", " ", 
    "roots", " ", "of", " ", "this"}], " ", "*)"}], "\[IndentingNewLine]", 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"f", "[", "x_", "]"}], ":=", 
     RowBox[{"N", "[", 
      RowBox[{
       SuperscriptBox["x", "3"], "-", 
       RowBox[{"2", " ", 
        SuperscriptBox["x", "2"]}], "+", "1.5"}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"set", " ", "the", " ", "number", " ", "of", " ", "iterations"}], 
    " ", "*)"}], "\n", 
   RowBox[{
    RowBox[{"n", "=", "7"}], ";"}], "\n", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Establish", " ", "the", " ", "domain", " ", "of", " ", "interest", " ", 
     "for", " ", "the", " ", "funciton", " ", "f"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"This", " ", "is", " ", "used", " ", "for", " ", "plotting"}], 
    " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"a", "=", 
     RowBox[{"-", "1.3"}]}], ";"}], "\n", 
   RowBox[{
    RowBox[{"b", "=", "3"}], ";"}], "\n", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Create", " ", "a", " ", "guess", "  ", "x"}], "-", 
     RowBox[{"value", " ", "for", " ", "the", " ", 
      RowBox[{"root", "'"}], "s", " ", "location"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"xold", "=", "1.4"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "a", " ", "table", " ", "of", " ", "2", "n", " ", "points",
      " ", 
     RowBox[{"(", 
      RowBox[{"0", ",", "0"}], ")"}]}], " ", "*)"}], "\n", 
   RowBox[{
    RowBox[{"points", "=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0.0", ",", "0.0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", 
         RowBox[{"2", " ", "n"}]}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Newton", "'"}], "s", " ", "method"}], " ", "*)"}], "\n", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Do", "[", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"xnew", "=", 
        RowBox[{"xold", "-", 
         FractionBox[
          RowBox[{"f", "[", "xold", "]"}], 
          RowBox[{
           SuperscriptBox["f", "\[Prime]",
            MultilineFunction->None], "[", "xold", "]"}]]}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"ynew", "=", 
        RowBox[{"f", "[", "xnew", "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"xold", "=", "xnew"}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"points", "\[LeftDoubleBracket]", 
         RowBox[{
          RowBox[{
           RowBox[{"2", " ", "i"}], "-", "1"}], ",", "1"}], 
         "\[RightDoubleBracket]"}], "=", "xnew"}], ";", "\[IndentingNewLine]",
        
       RowBox[{
        RowBox[{"points", "\[LeftDoubleBracket]", 
         RowBox[{
          RowBox[{
           RowBox[{"2", " ", "i"}], "-", "1"}], ",", "2"}], 
         "\[RightDoubleBracket]"}], "=", "0"}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"points", "\[LeftDoubleBracket]", 
         RowBox[{
          RowBox[{"2", " ", "i"}], ",", "1"}], "\[RightDoubleBracket]"}], "=",
         "xnew"}], ";", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"points", "\[LeftDoubleBracket]", 
         RowBox[{
          RowBox[{"2", " ", "i"}], ",", "2"}], "\[RightDoubleBracket]"}], "=",
         "ynew"}]}], ",", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}], ";"}], "\n", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
     "Plot", " ", "the", " ", "input", " ", "function", " ", "and", " ", 
      "the", " ", "line", " ", "y"}], "=", "0"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"fctplot", "=", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"f", "[", "x", "]"}], ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "a", ",", "b"}], "}"}], ",", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Thickness", "[", "0.01", "]"}], ",", 
          RowBox[{"Thickness", "[", "0.001", "]"}]}], "}"}]}], ",", 
       RowBox[{"Axes", "\[Rule]", "False"}], ",", " ", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Plot", " ", "the", " ", "points", " ", "connected", " ", "by", " ", 
     "line", " ", "segments"}], " ", "*)"}], "\n", 
   RowBox[{
    RowBox[{"polyplot", "=", 
     RowBox[{"ListPlot", "[", 
      RowBox[{"points", ",", 
       RowBox[{"Joined", "\[Rule]", "True"}], ",", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"PointSize", "[", "0.03", "]"}], ",", " ", "Red"}], "}"}]}],
        ",", 
       RowBox[{"Axes", "\[Rule]", "False"}], ",", " ", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";"}], "\n", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Plot", " ", "the", "  ", "points"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"pointplot", "=", 
     RowBox[{"ListPlot", "[", 
      RowBox[{"points", ",", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"PointSize", "[", "0.03", "]"}], ",", " ", "Red"}], "}"}]}],
        ",", 
       RowBox[{"Axes", "\[Rule]", "False"}], ",", " ", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Put", " ", "all", " ", "the", " ", "plots", " ", "together"}], 
    " ", "*)"}], "\n", 
   RowBox[{"pic", "=", 
    RowBox[{"Show", "[", 
     RowBox[{"fctplot", ",", "pointplot", ",", "polyplot"}], "]"}]}], 
   "\[IndentingNewLine]", "\n", 
   RowBox[{"(*", " ", 
    RowBox[{"Output", " ", "to", " ", "a", " ", "postscript", " ", "file"}], 
    " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Export", "[", 
     RowBox[{"\"\<Newton_fails.eps\>\"", ",", "pic"}], "]"}], 
    ";"}]}]}]], "Input",
 CellChangeTimes->{{3.4263428242338576`*^9, 3.426342858212717*^9}, {
   3.426342946770056*^9, 3.426343160857899*^9}, {3.4263436648425937`*^9, 
   3.426343666755344*^9}, {3.4263437358947616`*^9, 3.4263437490136256`*^9}, 
   3.42634381336616*^9, {3.42634416953831*^9, 3.426344213631714*^9}, {
   3.4263442500040145`*^9, 3.4263442530584064`*^9}, {3.426344555112739*^9, 
   3.426344572888299*^9}, {3.426344740158822*^9, 3.4263447469185424`*^9}, {
   3.4263448441683807`*^9, 3.4263448496662865`*^9}, {3.4263448916466513`*^9, 
   3.426344907589576*^9}, {3.42634514978784*^9, 3.42634515479504*^9}, {
   3.426350278993275*^9, 3.426350285963298*^9}, {3.4263511504263344`*^9, 
   3.4263511717069345`*^9}, {3.4263512018502784`*^9, 3.4263512110334835`*^9}}],\


Cell[BoxData[
 GraphicsBox[{{{}, {}, 
    {Hue[0.67, 0.6, 0.6], Thickness[0.01], LineBox[CompressedData["
1:eJwVl3c81f8Xx5VVSDSUUVIqJSu+ss9NoSRZlU1SKDuplFJGVjYh2SGEUJTy
JiOyblx775l5h+33/v11H8/7eY/zfr3POe9zBCwcdO5sp6Ojq9tGR/f/32It
pS/19WQkN8ba1aHJUVaqUtZx5zcZpSUEsG4QOMoq5ZTXN8vJqFbS8v0JSY4y
oqDqBYkiMromYd/nd4CjbGL56t/IJDKard3zWKB8dxl3ksk/U1cyeviWdkZp
nL3MbeHZ8QV+MsreCEt2kN9VJh9WErn/wRLycUnQuELPUrZVVM+uvr6ApDhu
+b2PoC+7XhLapBc/j2yDbRWFHqyj6PGt0SdBs+i51Q8pKu8S2inAYCqQNo18
UkqWI++PoLj9jncezk2gRgf52oxtNSjEUmdP/vAEiuhm/h3J/Rt5F0ih2fYJ
JNp+Nevg2Wpkr71y0LpsAkkn5gb1W1YiePOq3jB0AsXfz4yw+1uGhhgiJQmS
E6h4dvlxc2ABEiJ/28b6aBw58vndNt+ZAAXN9PGJm6OIXSeXMPi8AbTur/23
b2kU6Q27aXBBI8zSLzW8Hh9FyY6wq3KzEYT+G9qwJ46isSVWfhUlIsS/LTNR
TMbzQ4bSh4r/gr+x+6EulVGkXeo54p9EAvMxatzeNyMokc7pFvv5LmBbnXzn
wzuMcqqs9t0zHYIoCf6xMo5hJFI/mRnnOQRHbPTE1xiHUYm7S21BxhBIdZRW
OswPoQdC8yXBi0Ng/DX8n37VEDL1i7qw5TUMWc6KIOwwhFIHNVUrUkdAfTpk
qOnXIEoWmzTdQxwDv55zp7jvDaAEs/gdVu1T8F+BBPGe6QASFFBj8F2egkE/
4Uc/dAbQUI5dtz/3NMie468ykx9AC/LB56WNpmEylOlWGtsA8nNNWhTsnYaw
oEwJK9N+RIB/e+91zwCn5ShH3EQv+pb1OJ9UPgvWnJ2m8LcX3TlL+e9C/yyU
ldZnD33rReYzv1HE+iw4chdeOhXQi0SY61gHpeeA2PjK46tIL7p2tsqfP3MO
QmSPzBOde9CZg3JC8SrzwLHbqIlxowtJZe8U4eVbgPOrtS+lx7sQa66whLjQ
AjiPykhZ/e1C/B0zXLJSC0Aq4Yqp+dCFWGafR53QWIBo62aLAM0uFG2uTHR8
ugACvy5TORI7keGhSy+r2hfg7EOZQ3wXOhBrj7GT/+tFuG2e3qQh0oH+zfxi
PBa2CBFXuF65H+hA37INKblxi0AVII/1TbcjIX9tStrnRfjemJuXFNGOtpN/
DO3sXgRloZMXT461obuEwOo04SXQ7d5//6xfK/JL5lG/WbkEbQa5S8QHrejf
mHiwXsMSGHZceuZg2oqqo40PqrUtgXnr0zfZkq1I102FkW1iCWybBnNP9JGQ
cVbkp/2sZPCp+ETmkSQhhSdLBsLXyFCcpfKcvrcZfbT8JfakiQzyp/uZkn83
ow/ne8aOt5OhNONxMCG/GR2UidSr6SND5YesRPfXzag/7T+L2X9k+JvAWUGT
aEau220yESsFpsJ7mWd8/qLWOyYcbioU4Hv2MJQkTkQN7rEEq88UGJaJbkni
JqL70rdHjYspkEn5vt9hOxFV/T5PUUUUcO6jk9qb0YRUdvtbL9dTILuSS8V9
thFV31c/9HecAkdCla20nzYgYZFUZMJLhRwW3+lpjQbUH3xPLk6ACopeDQ4+
hxtQxX92Nk0nqWD4SP9xSXk9Khx1pO6VokKEib3f8R31KJjjoQaXBhWYT8dm
rUT8QZ6W/hP0blSISukXi7j7B0Ud/FIX/IIKgoeOF4rK/EFX3ZSzObypQODI
+3G7uxYpDzY+Wgimghu1qqFBoBa9e/S5WOwDFWZ/Lcwl5fxG1tFFg8kNVGgz
uiylXlOJbBSj/npy0YBL2/vlVlwlEv/Y6uvIQ4ObquWNhU6VCOmdOaV3mAYd
4rI2/LyViD5/VZLuBA26GU/Fk+0q0D8Xnq5pKRoM5O7YkbD3F7oTa5KxT5sG
Rz5cvK43UY40r1XkOOnR4FasR/LOn+XoitmnzqqbNBjyWlZwuYu5fTJL15QG
owYTTpe/lSHJA/SZjPdp0MocUjT+sxTvn6NR/YoGNxaZdXSsSpE7M5vBW29s
T8+LmR8cpShG3KjKzJcGXZ8dBMJu/0QlZz7Wt73B9hhdC1DY+QPZ/ti+aRJN
g6lcdrNQvW8ouShFmJRFg/uxPitrG8Vogq+OzJdDgxmvzfC76cWoba+2gUke
DeYMZmvkV4vQ8w1d+qpCGpDpG8+OJXxFlblSF8R/0mDz5hsm+ekCJMNukP+k
ngYh8qdfm78pQNaN5b63G2kgwP+b2UesAImKVNRcINLg4hjdzr8P8lHur6fR
4y00CHBxYbVaz0P9Z2XWfnTRgE+fMzAwPg/xNnQmG/bQIEc+hy2fkIc0N6c7
Z3tpkMJI5F3dmYsaBTR3rg3QQPxo0ZETvNnIR8d9IXmMBrqT8wJOZlmoze9A
DGWcBq55p4/9SMlEL7RWKUqTNPipFH9c58xHxO7rPfRlGuvD2HHivWMGYqWl
PB2eoQFDA6fQRGE6Clug9jDP0kDdyPv0c8U0ZO99gvW/eRrYH0XCtS8/oC6+
J4OKCzQInVw+s686FW1TmAlTXMT38chWLFMzBeXsOhIrQKbBmlKaOCUsGWmU
3lhhpNDgMNOABKE9CbUuxqgOYr4ToSvVZpaI2oclSp/RaMBRxHNf+VICsolf
X1BYxv6nEmTXQXyPuDrRCTLmkVsPnRmGYtHiH940tVUanJ6fcIm9F4NkPU6v
DmN2em78SHzpLar/fMji0Rq+n9gLz4wZIhGTz3sPz3Ws/6mi54uB4ajyRCJh
HXNA0emXvvvD0BY8OGm7QYNmlXjPw/EhqIL4V7YFs0TGMTabq0GIp8P1mcQm
DUorWTgjTQPQnuuHyD6YNQYX9pc7+KIiVY8PLZg7Nzp4/nl4o/3ZhpEHtmhw
l6eMnzvsFZLxsqrRwTxmp3nZX/UFOtN464I3Zrp0Pt0h6ydow5eBNRczz8CU
sVzAA1RcwCJIxGxz1rlB0Og+sroqFj6BWe7yoV8fwQzxXBcxWcYc+FX/w0lB
QJ21yGcL82T3YJ+w0TXIz0zat4m5moFvhCfsNnymvmElYw4Ne0G8d9UBhrLB
cRCzaIiTqYnXQ9Cud1KrxrzvZooeo8VTYNPrj0zGvBF070NbggfwXbhg7orZ
0dkzMTPIE0Ki9bPOYx6+HvfuubsPOOt2PWXAfEP2S5SOrR80OHu2lWI9avka
Q08YBUI+hb3MEfNyZMRvUAiGfAXvUV7MEa8/Pc9mD4X7m9ILpVjvM5Uh3LIv
wkCNLn7cAHMlnUth1Vw45DFLNczg+zJWvHlNxzwSOLuc0x5hJj+Rm+ojRkGL
AunRKr5vwSW6I8ufY+Bo7ei+CewPP8RGvnsdfQde7n6TOpj1bH9f5wyPg/l2
6ZovKzTwGn0TcNolAQIiIr/cxv7FHioTbuScCCEv5RpysD8aimUOJJ1OArXT
RLpFKg0W7795KvouGZwqCxvMsf8qsWzVuOimwHcCzSEQ+7t/hiNXCWsqmARx
EvKWaHB0VPez2rMPYFBQ+HQAx4uOKc+4uUkGVJcw9Q3/o0H8eoBU+v6P8NTQ
wPYvjr+p2I2X/xo+QidFW6MIx+er9gE+N6UsABm+vbY4fvO10nXC+XNAzfOu
rNsI9u/Zg4md7TkwHS/XfHoYx+sb/3/8IbkQMPRqoGWQBgsj0hdveOTBHdoN
h939NPAT4Okfmv8MfHOmLWc6aZA738zIdDUfpIeORya143yLAs6c+pgP74TE
5tnbcP4y3XjicKsAFirnPFuaafAtdmDf+t9CsGlxS2DA+a7fJkaeX/QLuC9/
yFD6QwNGWR0LZf8vIBI3aO9UQwPt9opcX+WvwPLQ6WJ5JfbPvenq+wuKQJ9B
JZQF50/uN3YeohHfgf3GpTi1bJy/LWX38019B+PCgd3SmTSIU2DM3EkoAX/F
LMfDGfg9mYlrGZkugWUj4x29KTQ4caX+xLsLP2EzupXC+Y4GUjuFG5iXEMiH
G1JN8Hvw1HEtz02wDCSpJfMz+L0wNr711OlaGXz7fGePqycNDkmJcJh9KIPe
nwe6XJ/TIHG4UlZOuxxcV+MGzrrQIF2ZHDif8QvMvSxob/H75CtqeHO85Rd0
V89x1xnRwJqnTKBv8xeEBne/X9bH+Wgx4GudXgUcBr0OZV2cv5OODaTRVULW
mKN7hBoNCul0JU30q2BH+5svYmI0KC/93FnLXAM/ZVjUzFapwCPKKHFbugY+
upvxxVKp4Pxe33fdsgYsd5n1NS7i9/7ZlrTYrxp4qzQid2KaCj6yVyMintXC
KuFbe2A3FTQKJzTNFv7AnaAjOYol+H3/yF9J7mqAgRA3OU1XKrj+UW3f3f4X
1E3YpUV7KHD54sPDTTN/oeNjTbx5O66HSlPuBG1vhpk8z9KgZgpUfKYjs4k2
A62D7WxfDQU4Yr5zsHg3A3l5UEf9C66PrETVGSRbQKxE/bfhGwr0MXD9XA4i
QaNJXoerHAVUYDR5ULUd2Kri/O39yMB+TIqNUbgX3LSKGCT6FuEUt8PDzieD
YFHTFjW8ewEy7hi3ep4ZgccR4i87I2YhdVKA4Sb3OOzTuJosmjcF7vkvUtxJ
k5CWncNxQwD/zxngPvlyBrpNO/Yytg7DyXT7E89t5qDR17tEJmYAwqJI1ddd
5kCRwGy1LWgANr3lrERezIFX7znWv68GoM2SKaMnYg5mz3obedsOgM/R+FPy
ZXPw+vTMjDJhABYb73I9Y50H8ZyIIvHmftBSj+KWtZiH7PkXt8o9+oBJuXYy
1WoeRvoySSK2ffBDdv0bh9087BA9fSDzZh8InbIwnHg0D/XlIyrNon2wxSz6
7u0bPP/ly5mh3l7IqarioxbPQ9yjA457oBfYzlP4C7AOXoMf+vaw94DEZNX5
8P0LEOHLfo1ltRtuhEbdfsC7AIkJUV5cY92QOHguXfLkAiyz7w+wK+0GyVdu
ooVKuC/wl+M56NgNhhVbCl/sFkDma+qIQ3sXZKiwGBbVLcCz0W6V3K+d0PCv
6+nbv5iNpyUsP3TCYmTW+0e4byirj7kgHtEJCmMag+eGF4DgP3yM3bkTmnyC
rYtXF0CDM3ZmSLQTqL/3Pfp2ahEYODiUt33qgAvq/BEluM8IaWPkjipph4ir
dwgebxb/XzdSoz61w4hW1szF8EXwSiHa5iW0g9fNcyqN8YugkStoL+LVDtWW
16gDXxahzFbsm7ZmO1x68UKfaWQRfgyAmO54G0S/qmKom1yE+uw+zrGuNpj0
Zv0cPLcI84xVn8Ia28A/8O1O7rVFKFa4ZSD7tQ3qYnK/C+9ZAq2aD49KfdpA
s7CfT5uwBELfmoKKzrRBfNHx2v2qSzBynvPhrEAbzH6//7DryhJY17tLEQ60
QXA5reH2Tfy9gugjtb0N93O7X7jaLwHf5TNOFp2tuN6CwXdxS9Cz6zTvcGAr
pM54vzFPXoIj8Sr0RM9WIM/VyR7PWAJiqsNIh1srRFBvhuUULEHH4ninkk0r
tNE7Xij/swSC78/5mai1gv7hxA9jy0tQFlhRRNnRCmd1fi+rbS6B8Z1O3ZPb
W4HNZ/bKR3oy+LYogPMaCcr+KSzeZyeDAo+ukt4sCYR+diotHCODByXAVb2V
BNsX6MJ0TpGh5nDv8UtNJOgRFBotECXDEYnLMWa1JAgJdA1wlSVD3vcOpp6f
JFgx2tuxpkmGCcbUSLYMEpCC5YRN9Mgwz/tuszOZBDkVt56XGuD97d0Sy9+T
wEI4T/ClJRlkHujX9oeT4M+qhhOTG+7b4s0vGbwkQaqoS6XVCzIkKlpzrj4j
wXOLdwdqvcgQbXvc98tjEpz9M/kzIBjbK3jy2zNHEsTG+LBwfsD2JHL9unqL
BC4Nn0ycM/F+ljGhVSYk0NzWmteSS4Zl1gRPfUN8PptjN6O+k0E/N2YnURef
L049k4awPXS/vhdqkeAr0WlDv4oM1n0tNYVXSXBPpiyFl4j3n+GRZr5Egou2
49RnrWRwdPLMN1AhweFEdvW+LswiCtW/lUnQzGwynzhKhsdkhlmqIgmy5b0u
bJ8mg7neJf5ieRL4OGRF3Z4ng3ik8tFIWRLIt68onFjD+t7+1Z78Hwm4WAVC
XtNR4KDK0/QWSRLMK10anmCkQPFnbc8jZ7Fezg7S6rgvtdbtf+InjvVKi/LL
4qBANOlnDLsY1qvrZw8bFwXy+n0WckVIoM8+KmbPSwFxflKw3RmsnzKbZ9MR
CjgqgqeaMAnYXCXbxE9QgG70VpfiaRKMfTQ8FSZMAUJZa4rWKewvvS+fLYnj
9Z9Izb4QwnpzfmzSk6aAb8mBmrqTWG8V4tGv8ng9cW6CFGbNJ7SHB85ToONV
s+W3E9i/Ph2ufayK+1yT3otGmLcPqvB1XcH22bTMcmPu2WfnIK+N3wldTSfK
cRIcX5HYx3mTAo9PnPSfxWzfSy0eM6bApQcXCnbi8UXlJSY/LCjgsV95SxUz
XZrH9jBrCuyYLPdIw3zZXyXdyp4CNbW31IWwPWH2LBqKLli/8DL7eszdOk3z
e9woUHbUnS4cn0fwXETkxAsKDOhI73mGz2vHayBX6o31oAv4+hrr8XXrUH94
AF6fiYfuK9Zra3jI0yaUAsuEV9uYsb6XatKF4C0+T6BFg5soCUKzbRv2vaeA
TJWBKye+n64QCeepZHxf1Uy76yRIYGtQUhKZQ4GQsA2DdHzfXxQ9zO8X4vPu
95b4c44EmwIqjOe/U4BIVLzAIYfjbapRc6YK629easYEJOhsDF8qr6PAxExF
8NfzJDhaoB/99i9eX8D1ov9FEhQ+HRpU7sXzF4184tRJsGGW7nNwGO+n1iDT
jf1Z9aKt8OwE1rPw7ndFbRJ0sFFdYsgU0FJ589BeH49/v3PHPCsVhPR2tnpY
4/GvGrOrOKnQ4crLMmCL7bkbrv3uABXoQjVSbjvh/cUOvVM9RoXHvK81Otzw
+DJx0fdyVLj09qfnk2ASBKdSmp0JeLzXyeGNCLyf7/dHl1SpkKeWLp8ei+NL
+2L5kjYVyipuH7VJI0HQ0E09dRsqEOPEiVmIBO3VfCv8DlRIvPZxm2E1CY5k
Db6nuFBholi0XKSBBPkP7o8nelAh43C3lGo3CdoYXrjR3lJBX4VnWWEFx+fJ
tMTUaip4OJ2V1VFohQV5jRmueioUd/d6V19shUqtxXN+f7H9mUVBxldbwcZN
qcmuB8+PDeScNW2Fwoa2DWlcR3VUtnlm4Pys7sJsVHuIBmWalUa+La1wyO9T
mvwxGhzUmvXi72uF+fe6i5+EcN9yKcG4c6IVIn8n+IVJ4j5V9Yzel81WGOCR
KTa6TAOi/S42DvyeuJZb75/FdaO+V2W/yZs2uNzGfsvcjQbRGlSWk7FtwDdd
mN38ggZaCczte9PboHzftotF/ni9nWZB+uVtwGYd4+yRiPebXRMIobVBMvuf
pj24zl7eVX2o6l47NBkJ+587tgyXxo3s82w7oLFO1TxIaBkGpu9JFrh3QL28
hfSoyDJ0XDnG3BXUATW8MUNhMstAbE7ZG/G5A8q7meRnry6D+IydVslyBxQY
DU2nPl4G87qHrUNvOuGtUczVPY3L4ChGnFtp6YKousJjNi3LIMQVJRY72QUR
8sQV1LEMxevJquZbXRDCy5xmN7wMZe4fJE2Fu8Gv22WzZhl/h/boo97d8NRI
K/fFsRU4cl6RrQt6wMyImfPf4xUwFw1+YtXcC/VPy2SKn6/A/Ce7tCNzvSAX
98Tc02sFiCxvn+1g64N9vdO53CErwDH31c5StQ9qTImaqul4vEXKrWs/++Cs
RUxgQitmhjnPbYX9wGQjvEPn7Cpk1AoJZQ0NwAO/EbFDMqswbxC6Tk8egIGP
729OKK7CRLL25YeMg1AytTv9+eVVEMq45pgkNAiOtmSVLPNViPZwXzriOAhd
Dj89GYJXwdfBo6aSfgg+uV7b+jq9iuPn2LN+xWFI/T3ZcHhhFWTmrh5e0RqG
2INecT7UVbD+7wDrKcth8P1eLHtj2xocDDg2VOc/DJabAi4UrjUoXnW8frhz
GPh8yBOSymtwZORK3fqTEQiMjGnOi16DiTSb3y31o+A5Jpl0MH4NHl/O2F81
NApu5xodPFLWoKPY8V3D8ihYd27fpZWzBkQThfuCx8fg4iFbtbmKNVg2Y5ue
fD4Ga6lKP0Tm1oBOaX20S3oc7hUOf8hUWQdHIc843+IJIH7hrfa5sg7FifxH
fVsmQLpId8xCex0yWN8zp85OwPbvFSd4TdYhkZdyQOL4JMSilPQAl3XwaD57
Iyl8Emr/3P5om7IOZRyZfkMPp0BocDhbdNsGcLgWNJ/SmoHRXSMF+T82wDhQ
bfes6DzOeBtuwr82wIXen35NaR4qnbiUU39vgMfJlyb7r81DauslYmTzBqzr
TWo7O8yDxftPM48nNkDPUDCjIW8eBs48Og77NiGwyOTuLukF6LrC8rbu/iZY
bkthqVdfhCY/cbcRni0Q2kpa5osnwxQXqjt/ZAs0UphewmcyMKVePZRwHHMj
b5tDJa6bSu+VGYhvwUAvkbwyRYaMxVSmJpUtII5/sSmUocArI+7wb45bkFEz
bVTVRgEpke3ZQdVbcKaB+asjzhsaE5P2fLJ0hESDnYNF2A+DY5xIWkt0hA6f
xXrdi3QEFmN+Lp6kbQSP66cNPSbpCZZnPAPsTLYTyDkaV3xcdxBELMKu6XDR
E4gejlVTYbsIoh9/z9X9oifM9mz1KDzhJLBv5lc0uTMQHm9ra5wq2Uc4bln5
wPoMI4FLrYQ5//BBQpbYS+ewdkaC6esyOYVqXgLnTnKqaigTobNh9nIFxxFC
zlJgQbASM4GJcPTHcfWjhNMz3RSLVWYCX8HAxsqUIOGlWavNE8zJ53lWr04L
EjqaG3uDMRsfFqclYX79rbzqB+bo6vmlyzOChBGf9EiuNWbCnrkqWsw/QcJ7
gQf//cGs6lsmJDcvSOC4yfJQYoOZMMC3KfaUIkiw+kM/qYZ54kGFWhPmUsUN
Y1PMWnJ3zI9RBQl2gvMXAzCbcitF1GH+s9C6bxSzi6IKE9+yIMEzMKkwepOZ
kM2YuffnqiChaytWKBfzVKO0LOeaIOHsg4i4Ksx8BxPN7mAeMHjttYj5seiO
3F3rggSFk3Z6GlvMBEFfUX3TDUFCeOzdGgvMy3YJXvmYp3aZKzzBzNIz85lp
U5AQQ9YRTPs/Xz+0OxfzvJVG9A/MOyzogH5LkHCpW4WtBbP9GnK4iTlBEzwm
McsdNU7KwkwtlyFvYXZrJ7VsYf4fRp8Sjw==
      "]]}, 
    {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.001], 
     LineBox[CompressedData["
1:eJxN0H0s1HEcB/DL49h52rCdzO3mKZEwTUVUuqtMSJraeWg3l2hHdTvrzBXT
mJCd2JDkykMPq5ZTKqauCJc55DqPG3MXvxDnjknurtrq8/1+tvfee/3z/uND
Y2XGso1IJFLUn/zt1zGhL/v7te9I/66T/n6U3YPctf/wll6CPOjGCPdvQ57f
ODFUKUKmiBKXkrKQs9U57moqcnB5e6UDVwM2tPVbR2ypwafbhbK4uhVw1ZxB
xb/1A2xBM0miNS2Aax0usXnL82DxsHFdvV4FJm8Sdwq2z4KLJoO8KOnTYFsb
psxUNw4+NeFwMaBIDnbO4QlH/AbBX5nHAyN6u8D6+FKz4AUxuM95QOjBLAmD
f5Rycn0r3oIlnS/G+sx7wVlShsJGMQSmh6nuzzAUYGvXQLKp9xTYi5LJG+PP
gB+yE+T5PkpwA0EziafMgQUt1x8IRgiwvV2xgMhbBHs2Z3hcS1sGkw+tUcU2
anB4BLWivXAVfMalvvHbhgY8ac/JDD65BnbxbKpv+LQOljG9bwa5boCTmeZ2
S1d/gp9mRRteLWyC01tnGx/Tt8AqK6W4pUOH9or8spVOBnDkPJHhvI908L/L
qi+PxGiQLROojk6ibeAUn/xiTqIReBerPDrW0Rjs+6hn+fMHZGt9y0eZwATs
ntLFveBjCn6yO+9KuQLZzkLbwBCagZ9pSsRloebgnYsTa6xN5LxkeRof8+jw
wFQZ5sI3ku4OzMqC5krHX8h3adw9Usy28ZY8fx1yqtSYOIq584AuIQkzx23l
SDFmqVpur8KcXyJqrdIjjxtqdjzHHMCtqO3GPH228MYq5hBPTlykAfl2zfle
FubvVudC+JirtbFuTZhXUiOrOjAfm6CTv2C+FxWWS2Bel+zVGjD/Bns/PwY=

      "]]}}, {{}, 
    {RGBColor[1, 0, 0], PointSize[0.03], 
     PointBox[{{0.24285714285714066`, 0.}, {0.24285714285714066`, 
      1.3963644314868822`}, {2.0004183332721857`, 0.}, {2.0004183332721857`, 
      1.5016740331728577`}, {1.6253137116997287`, 0.}, {1.6253137116997287`, 
      0.5102119667971392}, {1.266938040278121, 0.}, {1.266938040278121, 
      0.3233387920043702}, {2.5482175720374443`, 0.}, {2.5482175720374443`, 
      5.059802996404999}, {2.003412742019388, 0.}, {2.003412742019388, 
      1.513697595057467}, {1.6275570236015322`, 0.}, {1.6275570236015322`, 
      0.5134202074650789}}]}, {}}, {{}, {}, 
    {RGBColor[1, 0, 0], PointSize[0.03], 
     LineBox[{{0.24285714285714066`, 0.}, {0.24285714285714066`, 
      1.3963644314868822`}, {2.0004183332721857`, 0.}, {2.0004183332721857`, 
      1.5016740331728577`}, {1.6253137116997287`, 0.}, {1.6253137116997287`, 
      0.5102119667971392}, {1.266938040278121, 0.}, {1.266938040278121, 
      0.3233387920043702}, {2.5482175720374443`, 0.}, {2.5482175720374443`, 
      5.059802996404999}, {2.003412742019388, 0.}, {2.003412742019388, 
      1.513697595057467}, {1.6275570236015322`, 0.}, {1.6275570236015322`, 
      0.5134202074650789}}]}}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  AxesOrigin->{0, 0},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{
  3.4263438141673117`*^9, {3.4263441723122993`*^9, 3.4263441971780543`*^9}, 
   3.426344256773749*^9, {3.4263445590984707`*^9, 3.426344573579293*^9}, 
   3.4263447509543457`*^9, 3.426350286744421*^9, 3.426351172728403*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
},
WindowSize->{793, 838},
WindowMargins->{{10, Automatic}, {Automatic, 4}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 1048, 27, 396, "Input"],
Cell[CellGroupData[{
Cell[1641, 52, 7559, 189, 1256, "Input"],
Cell[9203, 243, 5336, 94, 310, "Output"]
}, Open  ]],
Cell[14554, 340, 161, 5, 69, "Input"],
Cell[CellGroupData[{
Cell[14740, 349, 7325, 183, 1212, "Input"],
Cell[22070, 535, 20494, 341, 310, 11970, 200, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
