(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     45819,        946]
NotebookOptionsPosition[     45286,        926]
NotebookOutlinePosition[     45704,        942]
CellTagsIndexPosition[     45661,        939]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell["\<\

Mathematical Principles for Scientific Computing and Visualization  
http://www.farinhansford.com/books/scv
by Gerald Farin and Dianne Hansford

This notebook demonstrates polynomial least squares approximation with \
B-splines.
Figure 8.12 was generated with this notebook.

Compare the results here with those in leastsq_pol.nb

Input: Knot sequence called u. Data values (v_i,f(v_i))_i i=1,n ... Here we \
have selected 20 values from f(v) = sqrt(v) + sin(v)/10.  The v_i live within \
the u-knot vector.

Output: the least squares approximating B-spline to the input values.
Here we are using a degree 3 B-spline, and we are using five polynomial \
segments.
(See comments below -- you can add more segments by adding knots and see the \
improvement in the approximation.)


See Chapter 8: Data Fitting for details.

Updated July 2008\
\>", "Input",
 CellChangeTimes->{{3.425997898301854*^9, 3.4259979434567842`*^9}, {
  3.426004883144433*^9, 3.4260050369756317`*^9}, {3.42601112959639*^9, 
  3.426011156214665*^9}, {3.4260130001761513`*^9, 3.4260130157986155`*^9}},
 FormatType->"TextForm",
 Magnification->1.25],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{"Define", " ", "the", " ", "output", " ", "directory"}], " ", 
   "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"SetDirectory", "[", 
      RowBox[{"ToFileName", "[", 
       RowBox[{"Extract", "[", 
        RowBox[{
         RowBox[{"\"\<FileName\>\"", "/.", "\[InvisibleSpace]", 
          RowBox[{"NotebookInformation", "[", 
           RowBox[{"EvaluationNotebook", "[", "]"}], "]"}]}], ",", 
         RowBox[{"{", "1", "}"}], ",", "FrontEnd`FileName"}], "]"}], "]"}], 
      "]"}], ";"}], ")"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Set", " ", "a", " ", "zero", " ", 
     RowBox[{"tolerance", " ", "--"}], " ", "anything", " ", "smaller", " ", 
     "than", " ", "this", " ", "is", " ", "considered", " ", "to", " ", "be", 
     " ", "zero"}], " ", "*)"}], "\[IndentingNewLine]", " ", 
   RowBox[{
    RowBox[{"eps", "=", 
     RowBox[{"5.0", "^", 
      RowBox[{"-", "6"}]}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Definition", " ", "of", " ", "a", " ", "B"}], "-", "spline"}], 
    " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "It", " ", "is", " ", "composed", " ", "of", " ", "a", " ", "left", " ", 
     "and", " ", "right", " ", "span"}], " ", "*)"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"spanleft", "[", 
      RowBox[{"n_", ",", "knots_", ",", "i_", ",", "x_"}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Abs", "[", 
         RowBox[{
          RowBox[{"knots", "\[LeftDoubleBracket]", 
           RowBox[{"i", "+", "n", "+", "1"}], "\[RightDoubleBracket]"}], "-", 
          
          RowBox[{"knots", "\[LeftDoubleBracket]", 
           RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}]}], "]"}], "<", 
        "eps"}], ",", "0", ",", 
       FractionBox[
        RowBox[{"x", "-", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}]}], 
        RowBox[{
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "n", "+", "1"}], "\[RightDoubleBracket]"}], "-", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}]}]]}], "]"}]}], 
    ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"spanright", "[", 
      RowBox[{"n_", ",", "knots_", ",", "i_", ",", "x_"}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Abs", "[", 
         RowBox[{
          RowBox[{"knots", "\[LeftDoubleBracket]", 
           RowBox[{"i", "+", "n", "+", "2"}], "\[RightDoubleBracket]"}], "-", 
          
          RowBox[{"knots", "\[LeftDoubleBracket]", 
           RowBox[{"i", "+", "2"}], "\[RightDoubleBracket]"}]}], "]"}], "<", 
        "eps"}], ",", "0", ",", 
       FractionBox[
        RowBox[{
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "n", "+", "2"}], "\[RightDoubleBracket]"}], "-", 
         "x"}], 
        RowBox[{
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "n", "+", "2"}], "\[RightDoubleBracket]"}], "-", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "2"}], "\[RightDoubleBracket]"}]}]]}], "]"}]}], 
    ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Recursive", " ", "definition", " ", "of", " ", "a", " ", "B"}], 
     "-", "spline"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "First", " ", "procedure", " ", "is", " ", "the", " ", "terminating", " ",
      "condition"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"bspline", "[", 
      RowBox[{"n_", ",", "knots_", ",", "i_", ",", "x_"}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "\[GreaterEqual]", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}]}], "&&", 
        RowBox[{"x", "<", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "2"}], "\[RightDoubleBracket]"}]}]}], ",", "1", 
       ",", "0"}], "]"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"bspline", "[", 
      RowBox[{"n_", ",", "knots_", ",", "i_", ",", "x_"}], "]"}], ":=", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"spanleft", "[", 
         RowBox[{"n", ",", "knots", ",", "i", ",", "x"}], "]"}], " ", 
        RowBox[{"bspline", "[", 
         RowBox[{
          RowBox[{"n", "-", "1"}], ",", "knots", ",", "i", ",", "x"}], 
         "]"}]}], "+", 
       RowBox[{
        RowBox[{"spanright", "[", 
         RowBox[{"n", ",", "knots", ",", "i", ",", "x"}], "]"}], " ", 
        RowBox[{"bspline", "[", 
         RowBox[{
          RowBox[{"n", "-", "1"}], ",", "knots", ",", 
          RowBox[{"i", "+", "1"}], ",", "x"}], "]"}]}]}], "/;", 
      RowBox[{"n", ">", "0"}]}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Define", " ", "knots", " ", "for", " ", "a", " ", "B"}], "-", 
     "spline"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "The", " ", "last", " ", "knot", " ", "is", " ", "made", " ", "a", " ", 
     "little", " ", "larger", " ", "so", " ", "that", " ", "we", " ", 
     RowBox[{"won", "'"}], "t", " ", "get", " ", "a", " ", "zero", " ", 
     "divide", " ", "when", " ", "evaluating", " ", "at", " ", "the", " ", 
     "very", " ", "end", " ", "of", " ", "the", " ", "domain", " ", "of", " ",
      "definition"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
     "In", " ", "figure", " ", "8.12", " ", "we", " ", "add", " ", "knots", 
      " ", "2"}], ",", 
     RowBox[{
     "6", " ", "and", " ", "then", " ", "1", " ", "to", " ", "look", " ", 
      "at", " ", "an", " ", "improve", " ", "approximation"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"u", "=", 
     RowBox[{"{", 
      RowBox[{
      "0", ",", "0", ",", "0", ",", "0", ",", "4", ",", "8", ",", "12", ",", 
       "16", ",", "20", ",", "20", ",", "20", ",", "20.00001"}], "}"}]}], 
    ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Degree", " ", "=", " ", "3"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"n", "=", "3"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Where", " ", "we", " ", "will", " ", "evaluate", " ", "the", " ", 
     "function", " ", "f", " ", "to", " ", "create", " ", "data", " ", 
     "values"}], " ", "*)"}], " ", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"v", "=", 
     RowBox[{"{", 
      RowBox[{
      "0", ",", "1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6", ",", 
       "7", ",", "8", ",", "9", ",", "10", ",", "11", ",", "12", ",", "13", 
       ",", "14", ",", "15", ",", "16", ",", "17", ",", "18", ",", "19", ",", 
       "20"}], "}"}]}], ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"ldata", "=", 
     RowBox[{"Length", "[", "v", "]"}]}], ";"}], " ", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "The", " ", "number", " ", "of", " ", "parameter", " ", "intervals", " ", 
     "and", " ", "degree", " ", "determines", " ", "the", " ", "number", " ", 
     "of", " ", "control", " ", "points"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"k", "=", 
     RowBox[{
      RowBox[{"Length", "[", "u", "]"}], "-", "3"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Number", " ", "of", " ", "control", " ", "points", " ", "is", " ", "l"}],
     " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"l", "=", 
     RowBox[{"k", "-", "n", "+", "1"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"A", " ", "B"}], "-", 
     RowBox[{
     "spline", " ", "is", " ", "define", " ", "as", " ", "a", " ", "product", 
      " ", "of", " ", "b"}], "-", 
     RowBox[{
     "spline", " ", "polynomials", " ", "and", " ", "coefficients", " ", 
      "d_i"}]}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"bsfunct", "[", 
      RowBox[{"n_", ",", "u_", ",", "d_", ",", "l_", ",", "x_"}], "]"}], ":=",
      
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "0"}], "l"], 
      RowBox[{
       RowBox[{"d", "\[LeftDoubleBracket]", 
        RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}], " ", 
       RowBox[{"bspline", "[", 
        RowBox[{"n", ",", "u", ",", "i", ",", "x"}], "]"}]}]}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
     "Create", " ", "the", " ", "generalized", " ", "Vandermonde", " ", 
      "matrix", " ", "for", " ", "a", " ", "degree", " ", "3", " ", "B"}], 
     "-", "spline"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"The", " ", "B"}], "-", 
     RowBox[{
     "spline", " ", "functions", " ", "are", " ", "evaluated", " ", "at", " ",
       "each", " ", "knot"}]}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Vander", "[", 
      RowBox[{"n_", ",", "l_", ",", "u_", ",", "ldata_", ",", "v_"}], "]"}], ":=",
      
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"bspline", "[", 
        RowBox[{"3", ",", "u", ",", "i", ",", 
         RowBox[{
         "v", "\[LeftDoubleBracket]", "j", "\[RightDoubleBracket]"}]}], "]"}],
        ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", "ldata"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "0", ",", "l"}], "}"}]}], "]"}]}], ";"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Test", " ", "the", " ", "Vandermonde", " ", "matrix"}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Vander", "[", 
     RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], "  ", 
    "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Define", " ", "the", " ", "function", " ", "from", " ", "which", " ", 
     "to", " ", "take", " ", "values"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"f", "[", "x_", "]"}], ":=", 
     RowBox[{"N", "[", 
      RowBox[{
       SqrtBox["x"], "+", 
       FractionBox[
        RowBox[{"Sin", "[", "x", "]"}], "10"]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "matrix", " ", "part", " ", "of", "  ", "the", " ", 
     "normal", " ", "equations"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Mat", "[", 
      RowBox[{"n_", ",", "l_", ",", "u_", ",", "ldata_", ",", "v_"}], "]"}], ":=",
      
     RowBox[{
      RowBox[{"Transpose", "[", 
       RowBox[{"Vander", "[", 
        RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], 
       "]"}], ".", 
      RowBox[{"Vander", "[", 
       RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}]}]}], 
    ";"}], " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "data", " ", "by", " ", "evaluating", " ", "the", " ", 
     "function", " ", "at", " ", "the", " ", "evaluation", " ", "points"}], 
    " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"data", "[", "ldata_", "]"}], ":=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{"v", "\[LeftDoubleBracket]", "j", "\[RightDoubleBracket]"}], 
        "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", "ldata"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Create", " ", "the", " ", "b"}], "-", 
     RowBox[{"spline", " ", "coefficients"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"bvals", "[", 
      RowBox[{"n_", ",", "l_", ",", "u_", ",", "ldata_", ",", "v_"}], "]"}], ":=",
      
     RowBox[{
      RowBox[{"Inverse", "[", 
       RowBox[{"Mat", "[", 
        RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], 
       "]"}], ".", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"Transpose", "[", 
         RowBox[{"Vander", "[", 
          RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], 
         "]"}], ".", 
        RowBox[{"data", "[", "ldata", "]"}]}], ")"}]}]}], ";"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Load", " ", "the", " ", "b"}], "-", 
     RowBox[{"spline", " ", "coefficients", " ", "into", " ", "d"}]}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"d", "=", 
     RowBox[{"N", "[", 
      RowBox[{"bvals", "[", 
       RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], 
      "]"}]}], ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", " ", 
   RowBox[{
    RowBox[{"Print", "[", 
     RowBox[{"\"\<B-spline coeffs d: \>\"", ",", "d"}], "]"}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"Just", " ", "to", " ", "check"}], " ", "..."}], " ", "print",
        " ", "out", " ", "the", " ", "b"}], "-", 
      RowBox[{
      "spline", "  ", "at", " ", "a", " ", "given", " ", "parameter"}]}], " ",
      "=", " ", "4.4"}], " ", "*)"}], "\[IndentingNewLine]", " ", 
   RowBox[{"(*", " ", 
    RowBox[{"bsfunct", "[", 
     RowBox[{"n", ",", "u", ",", "d", ",", "l", ",", "4.4"}], "]"}], " ", 
    "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"PLT", "=", 
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"f", "[", "x", "]"}], ",", 
        RowBox[{"bsfunct", "[", 
         RowBox[{"n", ",", "u", ",", "d", ",", "l", ",", "x"}], "]"}]}], 
       "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"v", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
        ",", 
        RowBox[{
        "v", "\[LeftDoubleBracket]", "ldata", "\[RightDoubleBracket]"}]}], 
       "}"}], ",", 
      RowBox[{"PlotStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"Thickness", "[", "0.01`", "]"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"Thickness", "[", "0.01`", "]"}], "}"}]}], "}"}]}], ",", 
      RowBox[{"BaseStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"\"\<FontFamily\>\"", "\[Rule]", "\"\<Helvetica\>\""}], ",", 
         
         RowBox[{"\"\<FontSize\>\"", "\[Rule]", "11"}]}], "}"}]}]}], "]"}]}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Create", " ", "a", " ", "postscript", " ", "plot"}], " ", "*)"}],
    "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Export", "[", 
      RowBox[{"\"\<bspleast4.eps\>\"", ",", "PLT"}], "]"}], ";"}], 
    ")"}]}]}]], "Input",
 CellChangeTimes->{{3.4259980799730844`*^9, 3.4259980802735167`*^9}, {
   3.4259982914471693`*^9, 3.425998291667486*^9}, {3.4259985206467423`*^9, 
   3.4259988166223345`*^9}, {3.425998853935989*^9, 3.425998912870733*^9}, {
   3.425998955241659*^9, 3.4259990039216576`*^9}, {3.425999039612979*^9, 
   3.4259990998395805`*^9}, {3.4259991947760925`*^9, 
   3.4259992059621773`*^9}, {3.425999237827998*^9, 3.425999304503874*^9}, {
   3.4259993706389713`*^9, 3.425999380653371*^9}, {3.4259994132402287`*^9, 
   3.425999466246448*^9}, {3.4259995116216946`*^9, 3.425999731037198*^9}, 
   3.425999797903347*^9, {3.425999836999565*^9, 3.4259998761458545`*^9}, {
   3.425999915312173*^9, 3.425999928541195*^9}, {3.426000016327426*^9, 
   3.4260000203431997`*^9}, {3.426000217576808*^9, 3.426000231787242*^9}, {
   3.426000304031123*^9, 3.426000306865198*^9}, {3.42600035398295*^9, 
   3.4260003591804237`*^9}, {3.4260004130779247`*^9, 
   3.4260004452742205`*^9}, {3.426004171561227*^9, 3.426004232108289*^9}, {
   3.426004626946038*^9, 3.4260047297438536`*^9}, {3.426004796319585*^9, 
   3.426004843116876*^9}, {3.426013030459697*^9, 3.4260131592749243`*^9}},
 Magnification->1.25],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"B-spline coeffs d: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{
    "0.025469008053589225`", ",", "1.7363854773441818`", ",", 
     "1.6765443276312213`", ",", "3.1220208663170657`", ",", 
     "3.227228394866445`", ",", "4.373828442265859`", ",", 
     "3.8164277370229294`", ",", "5.077125755456395`"}], "}"}]}],
  SequenceForm[
  "B-spline coeffs d: ", {0.025469008053589225`, 1.7363854773441818`, 
   1.6765443276312213`, 3.1220208663170657`, 3.227228394866445, 
   4.373828442265859, 3.8164277370229294`, 5.077125755456395}],
  Editable->False]], "Print",
 CellChangeTimes->{{3.425998932008251*^9, 3.4259989680400624`*^9}, 
   3.4259993617862415`*^9, 3.4259997018652515`*^9, 3.4259997414321456`*^9, 
   3.4259997994155216`*^9, {3.4259998520311794`*^9, 3.425999899359234*^9}, 
   3.4260000215649567`*^9, 3.4260002183679457`*^9, 3.426000307826581*^9, {
   3.426004193753137*^9, 3.426004233510305*^9}, 3.426004734080089*^9, {
   3.4260047976815434`*^9, 3.426004813073676*^9}, 3.42600484572062*^9, 
   3.4260131664452343`*^9},
 Magnification->1.25],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], Thickness[0.01], LineBox[CompressedData["
1:eJwV1nc8Vf8fB/Br3YHuuSQqGTfXta97kbbPu4iSFBJlZVNGhEqaVEIRssoK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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.01], 
    LineBox[CompressedData["
1:eJwV13c8Vf8fB3B7hnONtGwh495z7rUq+rxDCk0ysjIqGiRFRAuRVaisQvpR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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  BaseStyle->{"FontFamily" -> "Helvetica", "FontSize" -> 11},
  PlotRange->{{0, 20}, {0., 5.077125369528477}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{{3.426004208874881*^9, 3.4260042366247835`*^9}, 
   3.4260047385264826`*^9, {3.426004800826065*^9, 3.4260048169893064`*^9}, 
   3.4260048488751564`*^9, 3.4260131694695835`*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
},
WindowSize->{893, 891},
WindowMargins->{{71, Automatic}, {Automatic, 1}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 1128, 31, 454, "Input"],
Cell[CellGroupData[{
Cell[1721, 56, 17015, 416, 2294, "Input"],
Cell[18739, 474, 1112, 21, 50, "Print"],
Cell[19854, 497, 25416, 426, 322, 16825, 284, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
