(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     29677,        559]
NotebookOptionsPosition[     29145,        539]
NotebookOutlinePosition[     29564,        555]
CellTagsIndexPosition[     29521,        552]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell["\<\

Mathematical Principles for Scientific Computing and Visualization  
http://www.farinhansford.com/books/scv
by Gerald Farin and Dianne Hansford


This notebook demonstrates the Taylor series approximation to the Exp \
function.
Two things to play with: 
1) The number of polynomial terms in the series approximation -- see \
'approxnum'
2) The range over which we plot Exp and its approximation, called series \
here.
As the range increases, the number of terms needed increases.


See Chapter 2: Computational Basics for details.
\
\>", "Input",
 CellChangeTimes->{{3.420404362974968*^9, 3.4204043953515234`*^9}, {
  3.420404824057973*^9, 3.420404939083371*^9}},
 FormatType->"TextForm"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"approxnum", " ", "=", " ", "20"}], ";"}], "\[IndentingNewLine]", 
   
   RowBox[{
    RowBox[{
     RowBox[{"series", "[", "x_", "]"}], ":=", 
     RowBox[{"Sum", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "^", "i"}], ")"}], "/", 
        RowBox[{"Factorial", "[", "i", "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "0", ",", "approxnum"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"min", " ", "=", " ", 
     RowBox[{"-", "5"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"max", " ", "=", " ", "8"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"series", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "min", ",", "max"}], "}"}]}], "]"}], 
   "\[IndentingNewLine]", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"Exp", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "min", ",", "max"}], "}"}]}], "]"}]}]}]], "Input",
 CellChangeTimes->{{3.4204044352288637`*^9, 3.4204048139935007`*^9}, 
   3.420404954826008*^9}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVlnc81f8Xx81s917XTgqRiKxQxvtQJCUjIxRCJatEJNlb07eSvSNCSkaS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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{-5, 8}, {0., 731.3121675500453}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{{3.4204044640703363`*^9, 3.420404507202357*^9}, 
   3.4204045398292723`*^9, {3.4204045861158285`*^9, 3.4204047109653535`*^9}, {
   3.420404743722456*^9, 3.4204047505222335`*^9}, {3.4204047888873997`*^9, 
   3.420404814594365*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVl3c40N8Xx+29R0ZGlBHZkqx7zETZe1ZGUkSkpOyVUvlWsglZRUVW5PNJ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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{-5, 8}, {0., 731.3126227942631}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{{3.4204044640703363`*^9, 3.420404507202357*^9}, 
   3.4204045398292723`*^9, {3.4204045861158285`*^9, 3.4204047109653535`*^9}, {
   3.420404743722456*^9, 3.4204047505222335`*^9}, {3.4204047888873997`*^9, 
   3.4204048146344223`*^9}},
 ImageCache->GraphicsData["CompressedBitmap", "\<\
eJzlm2tsVUUQx0+5twXFGP3mBz9U9IMJKG1BRZCHIC8BAQVBHkKpaKtAEYoi
RQUfLQqIgMhDpCIgAiZiTCAmGI0kKomRaCSKmhBMDJpISCTGkJhc+5/Z85jT
3TOn3HuiUWJPz5n725nd2dmZ3XPruNqm+rnza5sa6morRy6qXVjfULe4ckTj
onZRrszzyuo9z/u10sN9of3WXArt/xUacIHEfn8d/cp53IA+8apwzUNSF9z1
sXDVuHYJPqW7bm611TEZ/j1gkdlMUZe8vJC1/6u2i7vyzdVB77sEdz2Du7zF
TLw70hsOa8md6GEZ4U0pTfe2tK2xW6uJz8C1gSCttaqgBSnLJVjrGrfh4Oq0
zjrDymnTGVmkb25w19fC1cj+Rt21hvFCFD+Z0Uc8ktyQ41/+vKJtxxf48RbQ
NTfh6CenIMePdyVdSUUgK2dZ5ZkzvwdqvzIuyuE3bvzl3+4PXMqEWdGjSy2y
7im5tLJ/Sl8FXcvhKt99+CGmC7sR7g6mIE/Xis1r1xz9qHHB+/gBi9/fzpy+
r7wweODWHJ68/N6VK4747QqhOjErad39b3JZMfrK/59RllZG7qE17+Wnfnjk
RxE+FFXdT02Z/BaCDD/H6+a8A2nr1i3HgM0+fOg7NL7+9OlzpPey/4hfEkKp
h+WjmiJklB+9y6t++P43uHLpnt3H4dq21paP/UUO12Ma8Lxu44ZPwYC96uzZ
P64pwrQZka0oDGE3+Glq4YH9X2PS0YMzE8fvQo/83iAIEADeSxF1pG0kCS7x
FWBIGM6fw4dt9xsj12EY3gvuvoyha7duFy785at6t3nZB+gHAhL9GnXs859a
3Bom0LUrfAzfHV7SdMgfQf17B09QB55xN59E1woYRx72DftNn3a3nMItYdhv
CeNwGYaz3N1yGs8QBga/oSU0UDJfFnf0DIavOH/+AjqFWIGHaF0vdduYRdc8
5sCvLWjymLtFLbcA5bfod+KbX5rcLerCFp811B9EtzCGRe4Wc8NeIVowvfDV
QneLB7kFKEwvhg+/NcbdVM8c5gyzAL/CZfPdmh9mGeYLvcdYvXlu/BGWYTXA
NYhUI7Pi88J+Yy2gfqM/De4WC1iGVI3+0LKrd+ONYX+AIwy9h+JeYcfmYBkQ
PGc8atX5KMvQaT+UzZRZ8cUsg68JDWOCw5j2wnQSoo1vLrirsChrsiibw8qa
2ZdeFPc1xw4rcVkhyMLS2hKWIWQDa7VZW4NfYZGmoYTW8vg8YgNzR/lhdmYj
elwGFgXrrMyscZ6j3S6SM6x690cCLRZeyYH2BMvgIDgKDvNmuoOcE3IO6x0L
mZ5muPHmMLCQ5ehpenxZNofew3go8U9361weLnXg1ONpbvxJlqFaoKzQ031u
/CmW+RWCnqaqOPqMFE6dmRofIJdNyq1Y0PQ0xa1zhaUL96q48EcCvpLCARsv
DeLlg2HRlEx247yjoOMcppqeJsW98CxDiDJa8ni6x63T4MjkGJaGP8cyTDGm
mp7uVnEUSBRRcloC/jzLEPJU6vA0UcWF9olxf7RYdE5w62yx6EzAWy3axzNO
Sz58L5ciSbRabBtlnBDKRSLzfxeT5FZZ+n8Xa8kgpa4KlzEGGLW2PIXaRFPl
+Nx4UCTCcSUyEBsLny9y2MGhGEVNcbKviHuuFB58kWVYqrT5wtPYSIh0qigZ
ZSKfjHXH+mqW4eRABxw8jVFwyoBRncSsDgMd2jRNayyzeqeKI/shC2r4WpYh
IIM3W6NVHOcCnCQ64ESvtQR6gk4+6MqcPUrFsWnHFj8ljnQCb9NGJgFfxzKR
4kequH+C64AT/bIldEa4dRoc/oWfU+JiKSbg61mGiktbSDwNV3HRGR0XnYni
RG+wxNwdbp0GF1lax8WUGJxSU64zSWIjy86NGd1GkYOnYays1CnVmBKnsWxN
idRnTIV5lJWlzuOBLSpF68NVB99RMTcWMqisr7BMZJCh2bjOmBLb7KEXG11G
magEQ92RvYllovzdruBUgqI6idlkWSMJml61rIIhKi4qlo6L456Oi+qSgG9m
mUg2g+P+2GxZe4NVnWL96DhcBxdq+BbLNA9ScZGhB8UHuMXSY12n6HECvtXS
44EqLhK6jou+67joexQnepulx7e5dW6z9FjHRY8Nbst+hVTJwigVySJBaZF5
7jXLshnAWsKjnf99vb1CdNaWWH1RW9YhXGRhMtEh8rexVXzps49LvF3IytZ2
luHLICq2eOpfbMgZpeIlUX932BNOBWeAAsXWRlQn0dstazjB8OuWYL1VxUW8
6bioTzouZl3HxU4pihO9gyGRrvq5dRpcHGJ0XLy91HExwAS8zRKZt6h48P1Q
HCfaQCIf6zrFAHVcDFDHMTgMUsPfYJk4gN0cH6CBcMDFQbcDZNcpQlTHxTsC
HReVx+Cd3/PuZJnYS/Jfr5V8r25MiWNBqUyVu8Iw27GIEDamSl5LdlriP2or
8iY2aqUEwxNryJhslsNLEWJvWlZYX3d4G1wcV6I40QYSL750naLK6Th6i15r
+C7LGuqj4iKl6rioHlGcaAOJjKPrFBlHx0XGScB3W/xRo+LCHzou/BHFid5t
8YeuU8SHjov4SMD3sEyU3GoVF7uL6vgA91hWia5TnEp1XLzU1XGRoQze+S8C
jDKx2TTKMvwug19cyJfCVaW3mo/MnSiEGdiyj1BUx6jV8G9Vii9ZxpYoj1WR
gMigZBmT4h2aMdn5kmWUiYRX5Y7/vZYF3ju+YvdalnVvVafYcuq4WOA6Ls7d
Oi6+R9BxsYQ7+ONthkSVuNGt0+CiSui4cJ+OC/fpuHCfjovYjOJRf4iYS9C5
zxJzN6i42ITpuCgvOi5yi46L6hzFiTaQqMm6TlGTdVzscA1u/t5JZJy0iWM/
y0QZ6RVVW/rEZ0yKt0LGZMn+tM54S7yiKaEN+4jEYszcmljLxlqG1dhYFRk1
Giid+stRo0ykjl7u2D/AMpE6esaXn4FEwuip6hR7Gh0Xm3gd9//3gZQ4BheU
lgT8ZHzsRf2vZl7Z36sc6mk=\
\>"]]
}, Open  ]]
},
WindowSize->{743, 747},
WindowMargins->{{Automatic, 75}, {Automatic, 58}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 699, 22, 264, "Input"],
Cell[CellGroupData[{
Cell[1292, 47, 1232, 36, 202, "Input"],
Cell[2527, 85, 11730, 198, 258, "Output"],
Cell[14260, 285, 14869, 251, 258, 11484, 194, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
