(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     42398,        965]
NotebookOptionsPosition[     41704,        940]
NotebookOutlinePosition[     42124,        956]
CellTagsIndexPosition[     42081,        953]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell["\<\

Mathematical Principles for Scientific Computing and Visualization  
http://www.farinhansford.com/books/scv
by Gerald Farin and Dianne Hansford

This notebook demonstrates polynomial least squares approximation (and the \
problems with it).
Figure 8.6 was generated with this notebook.

Input: 
Knot values u_i and function values there: (u_i,f(u_i))_i i=1,L ... Here we \
have selected L=20 values from f(u) = sqrt(u) + sin(u)/10. 

Once we have the data, we need to choose:
Degree n for the approximation. (Here we are using degre 11.)


Output: the least squares approximating polynomial to the input values using \
the monomial basis and then using the Bernstein basis. 


In the first cell below, we see that the degree 11 monomial polynomial \
approximates fairly well near the start of the approximation, but it deviates \
greatly from the input funtion f at the end of the approximation. This is \
typical behavior for the monomial basis.

In the second cell, the degree 11 polynomial from the Bernstein basis looks \
like a great fit over the entire knot interval. As stated in the text, the \
condition numbers of V^T*V -- normal equation matrix -- is 10^33 for the \
monomial basis and 10^6 for the Bernstein basis.


See Chapter 8: Data Fitting for details.

Updated July 2008\
\>", "Input",
 PageWidth->WindowWidth,
 CellChangeTimes->{{3.425938307500333*^9, 3.4259383576123905`*^9}, {
  3.4259386705824194`*^9, 3.42593867240504*^9}, {3.4259950035393906`*^9, 
  3.4259950195424013`*^9}, {3.4259950930681267`*^9, 3.4259952693916674`*^9}, {
  3.425995564105445*^9, 3.425995601529258*^9}, {3.4259957866153984`*^9, 
  3.4259958414242096`*^9}, {3.425997629275013*^9, 3.425997799750144*^9}, {
  3.4260113111774907`*^9, 3.4260113913728056`*^9}, {3.426011550772011*^9, 
  3.426011584861028*^9}, {3.4260118837508106`*^9, 3.4260119245094185`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"SetDirectory", "[", 
      RowBox[{"ToFileName", "[", 
       RowBox[{"Extract", "[", 
        RowBox[{
         RowBox[{"\"\<FileName\>\"", "/.", "\[InvisibleSpace]", 
          RowBox[{"NotebookInformation", "[", 
           RowBox[{"EvaluationNotebook", "[", "]"}], "]"}]}], ",", 
         RowBox[{"{", "1", "}"}], ",", "FrontEnd`FileName"}], "]"}], "]"}], 
      "]"}], ";"}], ")"}], " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   
   RowBox[{"(*", " ", 
    RowBox[{"===", 
     RowBox[{"=", " ", 
      RowBox[{
       RowBox[{
       "This", " ", "section", " ", "has", " ", "the", " ", "routines", " ", 
        "we", " ", "will", " ", "be", " ", "using"}], " ", "===", "==="}]}]}],
     " ", "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Create", " ", "a", " ", "monomial", " ", "design", " ", 
     RowBox[{"matrix", "  ", "--"}], " ", "see", " ", "Equation", " ", 
     RowBox[{"(", "8.4", ")"}], " ", "in", " ", "the", " ", "text"}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Rows", " ", "of", " ", "the", " ", "matrix", " ", "are", " ", "formed", 
     " ", "by", " ", "taking", " ", "powers", " ", "of", " ", "the", " ", 
     "parameters", " ", 
     RowBox[{"u", "[", "i", "]"}]}], " ", "*)"}], " ", "\[IndentingNewLine]", 
   
   RowBox[{"(*", " ", 
    RowBox[{
    "These", " ", "are", " ", "the", " ", "monomial", " ", "polynomials", " ",
      "evaluated", " ", "at", " ", "the", " ", "input", " ", "knot", " ", 
     "values"}], " ", "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Vander", "[", 
      RowBox[{"n_", ",", "L_", ",", "u_"}], "]"}], ":=", 
     RowBox[{"Table", "[", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"u", "\[LeftDoubleBracket]", "i", "\[RightDoubleBracket]"}], 
        RowBox[{"j", "-", "1"}]], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "L"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", 
         RowBox[{"n", "+", "1"}]}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Define", " ", "a", " ", "function", " ", "from", " ", "which", " ", "to",
      " ", "take", " ", "values"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"f", "[", "x_", "]"}], ":=", 
     RowBox[{
      SqrtBox["x"], "+", 
      FractionBox[
       RowBox[{"Sin", "[", "x", "]"}], "10"]}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Define", " ", "a", " ", "degree", " ", "n", " ", "polynomial", " ", "in",
      " ", "monomial", " ", "form", " ", "with", " ", "coefficients", " ", 
     RowBox[{"A", "[", "j", "]"}]}], " ", "*)"}], "\[IndentingNewLine]", " ", 
   
   RowBox[{
    RowBox[{
     RowBox[{"poly", "[", 
      RowBox[{"n_", ",", "A_", ",", "x_"}], "]"}], ":=", 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"j", "=", "1"}], 
       RowBox[{"n", "+", "1"}]], 
      RowBox[{
       RowBox[{"A", "\[LeftDoubleBracket]", "j", "\[RightDoubleBracket]"}], 
       " ", 
       SuperscriptBox["x", 
        RowBox[{"j", "-", "1"}]]}]}]}], ";"}], " ", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Define", " ", "a", " ", "matrix", "  ", "created", " ", "from", " ", 
     RowBox[{"V", "^", "T"}], " ", 
     RowBox[{"V", " ", "--"}], " ", "see", " ", "equation", " ", 
     RowBox[{"(", "8.5", ")"}]}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "This", " ", "is", " ", "the", " ", "matrix", " ", "part", " ", "of", " ",
      "the", " ", "normal", " ", "equations"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Mat", "[", 
      RowBox[{"n_", ",", "L_", ",", "u_"}], "]"}], ":=", 
     RowBox[{
      RowBox[{"Transpose", "[", 
       RowBox[{"Vander", "[", 
        RowBox[{"n", ",", "L", ",", "u"}], "]"}], "]"}], ".", 
      RowBox[{"Vander", "[", 
       RowBox[{"n", ",", "L", ",", "u"}], "]"}]}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"===", 
     RowBox[{"=", " ", 
      RowBox[{
      "End", " ", "of", " ", "the", " ", "routines", " ", "section"}]}]}], 
    " ", "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
     "Create", " ", "data", " ", "by", " ", "evaluating", " ", "the", " ", 
      "function", " ", "at", " ", "the", " ", "given"}], ",", " ", "L", ",", 
     "  ", 
     RowBox[{"knot", " ", "values", " ", "u_i"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"data", "[", "L_", "]"}], ":=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{"u", "\[LeftDoubleBracket]", "i", "\[RightDoubleBracket]"}], 
        "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "L"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Find", " ", "the", " ", "coefficients", " ", "of", " ", "the", " ", 
     "polynomial", " ", "by", " ", "solving", " ", "the", " ", "normal", " ", 
     "equations"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"This", " ", "is", " ", "solving", " ", "equation", " ", 
     RowBox[{"(", "8.5", ")"}]}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Coeffs", "[", 
      RowBox[{"n_", ",", "L_", ",", "u_"}], "]"}], ":=", 
     RowBox[{
      RowBox[{"Inverse", "[", 
       RowBox[{"Mat", "[", 
        RowBox[{"n", ",", "L", ",", "u"}], "]"}], "]"}], ".", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"Transpose", "[", 
         RowBox[{"Vander", "[", 
          RowBox[{"n", ",", "L", ",", "u"}], "]"}], "]"}], ".", 
        RowBox[{"data", "[", "L", "]"}]}], ")"}]}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"===", 
     RowBox[{"=", " ", 
      RowBox[{
       RowBox[{
        RowBox[{
        "Below", "  ", "we", " ", "construct", " ", "our", " ", "data", " ", 
         "and", " ", "solve", " ", "the", " ", "problem"}], " ", "==="}], 
       "="}]}]}], " ", "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Create", " ", "20", " ", "data", " ", "values"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"L", "=", "20"}], ";"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Degree", " ", "11", " ", "polynomial"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"n", "=", "11"}], ";"}], " ", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "uniform", " ", "knot", " ", "values", " ", "u_i"}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{"u", "=", 
    RowBox[{"Table", "[", 
     RowBox[{"i", ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "1", ",", "L"}], "}"}]}], "]"}]}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "the", " ", "degree", " ", "11", " ", "approximation"}], 
    " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "The", " ", "result", " ", "is", " ", "a", " ", "set", " ", "of", " ", 
     "coefficients", " ", "defining", " ", "the", " ", "degree", " ", "11", 
     " ", "least", " ", "squares", " ", "polynomial"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "We", " ", "are", " ", "solving", " ", "the", " ", "normal", " ", 
     "equations"}], " ", "*)"}], "\[IndentingNewLine]", " ", 
   RowBox[{
    RowBox[{"A", "=", 
     RowBox[{"N", "[", 
      RowBox[{"Coeffs", "[", 
       RowBox[{"n", ",", "L", ",", "u"}], "]"}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Plot", " ", "the", " ", "function", " ", "and", " ", "the", " ", 
     "degree", " ", "11", " ", "polynomial", " ", "approximation"}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"The", " ", "plot", " ", "is", " ", "over", " ", "x"}], "=", 
     RowBox[{
      RowBox[{"1", " ", "to", " ", "x"}], "=", "20"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"PLT", "=", 
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"f", "[", "x", "]"}], ",", 
        RowBox[{"poly", "[", 
         RowBox[{"n", ",", "A", ",", "x"}], "]"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
        ",", 
        RowBox[{"u", "\[LeftDoubleBracket]", "L", "\[RightDoubleBracket]"}]}],
        "}"}], ",", 
      RowBox[{"PlotStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"Thickness", "[", "0.01", "]"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"Thickness", "[", "0.01", "]"}], "}"}]}], "}"}]}], ",", 
      RowBox[{"BaseStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"\"\<FontFamily\>\"", "\[Rule]", "\"\<Helvetica\>\""}], ",", 
         
         RowBox[{"\"\<FontSize\>\"", "\[Rule]", "11"}]}], "}"}]}], ",", " ", 
      RowBox[{"PlotRange", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0", ",", "21"}], "}"}], ",", "Automatic"}], "}"}]}]}], 
     "]"}]}], "\[IndentingNewLine]", "\[IndentingNewLine]", " ", 
   RowBox[{"Export", "[", 
    RowBox[{"\"\<sqrt_approx11.eps\>\"", ",", "PLT"}], "]"}]}]}]], "Input",
 CellChangeTimes->{{3.4259383665752783`*^9, 3.4259384776349745`*^9}, {
  3.4259385108226957`*^9, 3.4259385911582127`*^9}, {3.4259387199333825`*^9, 
  3.4259387337131968`*^9}, {3.425938767321523*^9, 3.4259388407070465`*^9}, {
  3.425994694004301*^9, 3.4259948529027853`*^9}, {3.4259949214012814`*^9, 
  3.425994974036968*^9}, {3.4259956135665665`*^9, 3.425995637921587*^9}, {
  3.425995701102437*^9, 3.4259957662160654`*^9}, {3.425995876114091*^9, 
  3.4259958796291456`*^9}, {3.4259959152403517`*^9, 3.4259959227611666`*^9}, {
  3.42599598344843*^9, 3.4259960581057825`*^9}, {3.425996144620184*^9, 
  3.425996165720525*^9}, {3.4259962471275826`*^9, 3.4259963799886274`*^9}, {
  3.4259964450221405`*^9, 3.425996458872056*^9}, {3.4259967826275935`*^9, 
  3.4259967955261407`*^9}, {3.4259968297153025`*^9, 3.4259968368755984`*^9}, {
  3.425996939983861*^9, 3.4259969838869905`*^9}, {3.4259972112439137`*^9, 
  3.4259972195758944`*^9}, {3.4259972911287823`*^9, 3.425997298008675*^9}, {
  3.426011450277507*^9, 3.4260115220106535`*^9}, {3.426011738702241*^9, 
  3.426011752021393*^9}, {3.426011934183329*^9, 3.426011963165003*^9}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6", ",", "7", ",", "8", 
   ",", "9", ",", "10", ",", "11", ",", "12", ",", "13", ",", "14", ",", "15",
    ",", "16", ",", "17", ",", "18", ",", "19", ",", "20"}], "}"}]], "Output",\

 CellChangeTimes->{
  3.4259946582128353`*^9, {3.42599470069392*^9, 3.425994713161848*^9}, {
   3.425994770263957*^9, 3.4259948552361407`*^9}, 3.4259949749182353`*^9, {
   3.425996785021035*^9, 3.4259967962471776`*^9}, 3.426011968032001*^9}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], Thickness[0.01], LineBox[CompressedData["
1:eJwV13k8VG0bB3BbZkbMGRXZZ5ghMTNmULZyfpUKRZZSoZCUKCqFRNkq8Ygi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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.01], 
    LineBox[CompressedData["
1:eJwd1Hk41NsbAPCxzYbsU1mHYRBZruxu573SYpcWS0oRIb+oEFkiEk1KEqHF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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 2.},
  BaseStyle->{"FontFamily" -> "Helvetica", "FontSize" -> 11},
  PlotRange->{{0, 21}, {1.084147313308813, 9.510038512664323}},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{
  3.4259946582128353`*^9, {3.42599470069392*^9, 3.425994713161848*^9}, {
   3.425994770263957*^9, 3.4259948552361407`*^9}, 3.4259949749182353`*^9, {
   3.425996785021035*^9, 3.4259967962471776`*^9}, 3.426011968352462*^9}],

Cell[BoxData["\<\"sqrt_approx11.eps\"\>"], "Output",
 CellChangeTimes->{
  3.4259946582128353`*^9, {3.42599470069392*^9, 3.425994713161848*^9}, {
   3.425994770263957*^9, 3.4259948552361407`*^9}, 3.4259949749182353`*^9, {
   3.425996785021035*^9, 3.4259967962471776`*^9}, 3.426011968432577*^9}]
}, Open  ]],

Cell["\<\

Using the Bernstein basis functions rather than the monomial basis functions.
Much better results!
\
\>", "Input",
 CellChangeTimes->{{3.4259965545796766`*^9, 3.4259965812980957`*^9}, {
  3.425997591460638*^9, 3.42599759861092*^9}},
 FormatType->"TextForm"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Recursive", " ", "definition", " ", "of", " ", "the", " ", "Bernstein", 
    " ", "basis", " ", "functions"}], " ", "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Bern", "[", 
      RowBox[{"n_", ",", "0", ",", "t_"}], "]"}], ":=", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "t"}], ")"}], "n"]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Bern", "[", 
      RowBox[{"n_", ",", "n_", ",", "t_"}], "]"}], ":=", 
     SuperscriptBox["t", "n"]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Bern", "[", 
      RowBox[{"n_", ",", "i_", ",", "t_"}], "]"}], ":=", 
     RowBox[{
      RowBox[{"Binomial", "[", 
       RowBox[{"n", ",", "i"}], "]"}], " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "-", "t"}], ")"}], 
       RowBox[{"n", "-", "i"}]], " ", 
      SuperscriptBox["t", "i"]}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{
     "Evaluate", " ", "a", " ", "set", " ", "of", " ", "Bernstein", " ", 
      "degree", " ", "n", " ", "polynomials", " ", "evaluated", " ", "at", 
      " ", "a", " ", "given", " ", "t"}], "-", "value"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Bernvec", "[", 
      RowBox[{"n_", ",", "t_"}], "]"}], ":=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Bern", "[", 
        RowBox[{"n", ",", "i", ",", "t"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "0", ",", "n"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "A", " ", "Bezier", " ", "polynomial", " ", "is", " ", "formed", " ", 
     "by", " ", "multiplying", " ", "the", " ", "coefficients", " ", "by", 
     " ", "the", " ", "basis", " ", "functions"}], " ", "*)"}], 
   "\[IndentingNewLine]", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"Bez", "[", 
      RowBox[{"b_", ",", "n_", ",", "t_"}], "]"}], ":=", 
     RowBox[{
      RowBox[{"Bernvec", "[", 
       RowBox[{"n", ",", "t"}], "]"}], ".", "b"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"20", " ", "knot", " ", "values"}], " ", "*)"}], 
   "\[IndentingNewLine]", " ", 
   RowBox[{
    RowBox[{"L", "=", "20"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Degree", " ", "11"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"n", "=", "11"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Create", " ", "uniform", " ", "knot", " ", "values"}], " ", 
    "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"u", "=", 
     RowBox[{"Table", "[", 
      RowBox[{"i", ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "L"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Form", " ", "the", " ", "design", " ", "matrix", " ", "using", " ", 
     "the", " ", "Bernstein", " ", "basis"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"See", " ", "just", " ", "after", " ", "eqn", " ", 
     RowBox[{
      RowBox[{"(", "8.7", ")"}], " ", "--"}], " ", "we", " ", "form", " ", 
     "a", " ", "row", " ", "by", " ", "evaluating", " ", "each", " ", 
     "Bernstein", " ", "function", " ", "at", " ", "a", " ", "given", " ", 
     "knot", " ", "vale"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"For", " ", "Bernstein", " ", "polynomials"}], ",", " ", 
     RowBox[{
     "the", " ", "knots", " ", "are", " ", "assumed", " ", "to", " ", "be", 
      " ", 
      RowBox[{"in", " ", "[", 
       RowBox[{"0", ",", "1"}], "]"}]}]}], " ", "*)"}], "\[IndentingNewLine]",
    
   RowBox[{
    RowBox[{
     RowBox[{"Vander", "[", 
      RowBox[{"n_", ",", "L_", ",", "u_"}], "]"}], ":=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Bern", "[", 
        RowBox[{"n", ",", 
         RowBox[{"j", "-", "1"}], ",", 
         FractionBox[
          RowBox[{
           RowBox[{
           "u", "\[LeftDoubleBracket]", "i", "\[RightDoubleBracket]"}], "-", 
           RowBox[{
           "u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}], 
          RowBox[{
           RowBox[{
           "u", "\[LeftDoubleBracket]", "L", "\[RightDoubleBracket]"}], "-", 
           RowBox[{
           "u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}]]}], 
        "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "L"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", 
         RowBox[{"n", "+", "1"}]}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Evaluate", " ", "the", " ", 
     RowBox[{"function", " ", "--"}], " ", "again", " ", "with", " ", "the", 
     " ", "knots", " ", "scaled", " ", 
     RowBox[{"to", " ", "[", 
      RowBox[{"0", ",", "1"}], "]"}]}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"x_", ",", "u_"}], "]"}], ":=", 
     RowBox[{
      SqrtBox[
       FractionBox[
        RowBox[{"x", "-", 
         RowBox[{
         "u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}], 
        RowBox[{
         RowBox[{"u", "\[LeftDoubleBracket]", "L", "\[RightDoubleBracket]"}], 
         "-", 
         RowBox[{
         "u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}]]], "+",
       
      RowBox[{"Sin", "[", 
       FractionBox[
        RowBox[{"x", "-", 
         RowBox[{
         "u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}], 
        RowBox[{
         RowBox[{"u", "\[LeftDoubleBracket]", "L", "\[RightDoubleBracket]"}], 
         "-", 
         RowBox[{
         "u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}]], 
       "]"}]}]}], ";"}], " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Form", " ", "the", " ", "matrix", " ", "part", " ", "of", " ", "the", 
     " ", "normal", " ", "equations"}], " ", "*)"}], "\[IndentingNewLine]", 
   " ", 
   RowBox[{
    RowBox[{
     RowBox[{"Mat", "[", 
      RowBox[{"n_", ",", "L_", ",", "u_"}], "]"}], ":=", 
     RowBox[{
      RowBox[{"Transpose", "[", 
       RowBox[{"Vander", "[", 
        RowBox[{"n", ",", "L", ",", "u"}], "]"}], "]"}], ".", 
      RowBox[{"Vander", "[", 
       RowBox[{"n", ",", "L", ",", "u"}], "]"}]}]}], ";"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "data", " ", "at", " ", "the", " ", "knot", " ", "values", 
     " ", "by", " ", "evaluating", " ", "our", " ", "given", " ", 
     "function"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"data", "[", "L_", "]"}], ":=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{
         RowBox[{"u", "\[LeftDoubleBracket]", "i", "\[RightDoubleBracket]"}], 
         ",", "u"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "L"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "the", " ", "coefficients", " ", "of", " ", "the", " ", 
     "approximating", " ", "polynomial", " ", "by", " ", "solving", " ", 
     "the", " ", "normal", " ", "equations"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"Coeffs", "[", 
      RowBox[{"n_", ",", "L_", ",", "u_"}], "]"}], ":=", 
     RowBox[{"N", "[", 
      RowBox[{
       RowBox[{"Inverse", "[", 
        RowBox[{"Mat", "[", 
         RowBox[{"n", ",", "L", ",", "u"}], "]"}], "]"}], ".", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"Transpose", "[", 
          RowBox[{"Vander", "[", 
           RowBox[{"n", ",", "L", ",", "u"}], "]"}], "]"}], ".", 
         RowBox[{"data", "[", "L", "]"}]}], ")"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Load", " ", "the", " ", "coefficients"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"A", "=", 
     RowBox[{"N", "[", 
      RowBox[{"Coeffs", "[", 
       RowBox[{"n", ",", "L", ",", "u"}], "]"}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"PLT", "=", 
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"f", "[", 
         RowBox[{"x", ",", "u"}], "]"}], ",", 
        RowBox[{"Bez", "[", 
         RowBox[{"A", ",", "n", ",", 
          FractionBox[
           RowBox[{"x", "-", 
            RowBox[{
            "u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}], 
           RowBox[{
            RowBox[{
            "u", "\[LeftDoubleBracket]", "L", "\[RightDoubleBracket]"}], "-", 
            
            RowBox[{
            "u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}]]}], 
         "]"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"u", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
        ",", 
        RowBox[{"u", "\[LeftDoubleBracket]", "L", "\[RightDoubleBracket]"}]}],
        "}"}], ",", 
      RowBox[{"PlotStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"Thickness", "[", "0.02", "]"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"Thickness", "[", "0.01", "]"}], "}"}]}], "}"}]}], ",", 
      RowBox[{"BaseStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"\"\<FontFamily\>\"", "\[Rule]", "\"\<Helvetica\>\""}], ",", 
         
         RowBox[{"\"\<FontSize\>\"", "\[Rule]", "11"}]}], "}"}]}]}], "]"}]}], 
   " ", "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Plot", " ", "a", " ", "postscript", " ", "file"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Export", "[", 
     RowBox[{"\"\<sqrt_approx_bez_11.eps\>\"", ",", "PLT"}], "]"}], " ", 
    ";"}]}]}]], "Input",
 CellChangeTimes->{{3.4259965885184784`*^9, 3.425996773444389*^9}, {
  3.425996803747963*^9, 3.42599688817937*^9}, {3.4259969185430307`*^9, 
  3.425996924361397*^9}, {3.4259969922490144`*^9, 3.425997201820363*^9}, {
  3.425997233115363*^9, 3.4259972826966577`*^9}, {3.4259973214423714`*^9, 
  3.425997392134021*^9}, {3.4259975781515007`*^9, 3.425997584620803*^9}, {
  3.4260121805976553`*^9, 3.426012318335713*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], Thickness[0.02], LineBox[CompressedData["
1:eJwVzns8FfYfx3E1t5nK+RUyzXG/RCccndza5xvNJS1+mjH9SEKOkeJgRzll
IfFzmR1OatJPNNkJy5ElQthZWg4xO6mM3E6KnYvOOY5z+H33x/vxejz/e5tF
JwfHblRTU6Pi/dPZIwbxampC0EuPHnZoNwdHYsj9CUMhnL9FTEuCMshc+F6n
c7cQ7uiGi46rKqG/9dnRa75CyAkX1NHUboBeNuEO/ZgQblsoiltLaiD8UNBa
aIYQrG3KCn77oxbqDEsCKSVCyC/j2U3O3wL3Rl2x+CH26X1N2683QA49wHt4
TAiKH5cXTYPYwDtQwGxaEsJGu0W9kXd3IOaF1t5EExHYJthNq9OboVj7g3Oz
WSKgcriDdMcW4I/sf9pbLoK2xR3fvRpuAYvqCyY1d0RAupbkaZvMgV/2rHVH
vhLBdorq393XW+F19IoG31MMipoik4+etwGl8++SgVUxDJxJ1b/91wO4kE+a
qt8qgaOFuqcTwzrgyZEk50v2EtDwa3Rk8zogamFh1PuoBOqvPK2cvt8JBYZz
Rp3tEuCOMH/mMrpg4szLm030ZWjtba91TOiBPOvHbUzZe3Db5aCrmdIHpmaF
amkEKUS1JC7W3OyDduPP/UPspVBl/k3Z4mgfLOkNj+sfk8K2pC4ik9IPoat8
VcWvUijVSO7/StoPtkMC78pyGRTtZygdqFx4TNceqiKvQKd+l1GV5QDoPvUV
3EpWgk/Adz0cIg8qos0pQ5eUsG5y87XciwdEufLiSrUSYp+kj1jG8cDZ4i7x
0JAStsdz9u5h8yCMviNU5KiCJ7MzddXGQ1BrJer3EKtAg+SWwqodgn2MqzeH
aOugLO+aUjYOQxsrfbfKcwO6J6UIwxpGgHTh82ClszqKqC7gb/P+EyqCC2Ld
qVpo6eQnuW4vxmHx23Hjyngd5PKK4WW7YQL2VpU+otRvQoGPU/alV0/CiYxY
wReJekgzze5LstVrMAvU7TF5Q0CkzUMnBy9OA994jqCdtxWZb9VY0FyegUss
f062lj4aK3ngq+U1B3xSsk5ctgG66xueGZ09D+YPuHyq8XYkmAk+yR0RQIz0
tlxVaYRmPuq3uK69ANXWXaU/uxojfjP77bmgt3AupeIlj7cDWZQ3xswXvoOg
JBmdEWCCSC41qY3sRTCJKCw5NE9E1Hl6f8TkEoizvjj7R4spKjjuLv+f4m94
xlUmbMsxQ3MBmzubvxfCRWIeJz/IHLVKi3KfW4qg6fCo9WdbLFC8+5UgiwYR
hMk9QggTFoj1xjstzlUMWfR4tlW1JbpReey+/QMxpGqnRyT/xwp5HJMkMNwk
MEuoGvjNzhqZ9rHq6N0SGBHnRyW8sUY+s/mowHMZPj1cfNf4vg0ivVTzlXUv
Q2YbLXJjri0qDrDmCtF7cJITyU7edoizPsX49t57iG2Z6U0m7ERrX+6MILhI
Qd/nVJZ0eCeiPYpKzfhRChVz/NK7xfZIAfIfBAYyYG6IiGIcdUA0jSzD90Uy
2LRcF1b28S7klVfT67Mqgw+Oa2+692YX0izzt5lMlgOtbveC2U8kxD/Ynycd
l0ObbK9Zb/xuNMGlhvh4rYDHawcdd2NHlBmYs5TWvAJhLqZXKGcdkSh2lHRZ
XwEcb1N1j2lHZNt3eYdVNn7xXO+TcH8ndMad48YXKEDx4sTBqnonlPGMtWHM
dxXcr/WGCLY4I/W3p9YW2avg+mtuWuhpZ3T18ahXzodKIH7j6jU16Ixog6Oa
ldFKODAve5eyh4zqtY8nhvcoweaREdu2nIyYBWEXVYYq8Av3859TklEfQ9Em
TlXBQfKHzWLsg866LBX2Id0Bg3XsqDSDdG2aCoIeBswYqsjIt2Gbiwl2qHng
eX/sl4NtTf7YcQshrWzs7I2VdTewczNPmKWukRFrM7fwcJoK8o5Y5p/H/rhl
gfoVdr7D7FLhP57X9IvB/u9EXEct9mWVrXomNtMrIXQM22Bm67k67FqdM0Vu
62R0gJWToMS+Ne20/Bm2A9nJTytdBfUd4vBg7LzIMat/YbNP0Wy/xs49oTNl
g93yLKP3B+xaa6PQI9icn1x33saW7ytwicS+l7NS2or9y4yEQMVup5yNHMRO
mWz+nYHdscWzfxw7cY96w2XshwKl/Tz270ZBl5jY3T2dZRJsZhEzphr70VXG
yjp2d+nw/gbs/wOVVT56
     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.01], 
    LineBox[CompressedData["
1:eJwV03k8FPgfBvAZ+m1WyChKWmOM+zYjofL5blLuJvIilpQjq0MYyhkd2Ei2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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{5., 0},
  BaseStyle->{"FontFamily" -> "Helvetica", "FontSize" -> 11},
  PlotRange->{{1, 20}, {0., 1.8414903186896963`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{3.4260123218006954`*^9}]
}, Open  ]]
},
WindowSize->{795, 792},
WindowMargins->{{Automatic, 240}, {33, Automatic}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 1860, 44, 669, "Input"],
Cell[CellGroupData[{
Cell[2453, 69, 10993, 267, 1513, "Input"],
Cell[13449, 338, 519, 10, 48, "Output"],
Cell[13971, 350, 11316, 192, 302, "Output"],
Cell[25290, 544, 294, 4, 48, "Output"]
}, Open  ]],
Cell[25599, 551, 268, 8, 107, "Input"],
Cell[CellGroupData[{
Cell[25892, 563, 10764, 282, 1394, "Input"],
Cell[36659, 847, 5029, 90, 324, "Output"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
