(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     56676,       1161]
NotebookOptionsPosition[     55981,       1136]
NotebookOutlinePosition[     56401,       1152]
CellTagsIndexPosition[     56358,       1149]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell["\<\

Mathematical Principles for Scientific Computing and Visualization  
http://www.farinhansford.com/books/scv
by Gerald Farin and Dianne Hansford

This notebook demonstrates interpolation with B-splines.
Figure 8.10 was generated with this notebook.

Compare the results here with those in Taylor.nb

Input: Knot sequence u. Data values (v_i,f(v_i))_i i=1,n ... Here we have \
selected 20 values from f(v) = sqrt(v) + sin(v)/10. 

Output: the least squares approximating B-spline to the input values.
Here we are using a degree 3 B-spline, and we are using five polynomial \
segments.
(See comments below -- you can add more segments by adding knots and see the \
improvement in the approximation.)


See Chapter 8: Data Fitting for details.

Updated July 2008\
\>", "Input",
 CellChangeTimes->{{3.426007295162745*^9, 3.4260073468170204`*^9}, {
  3.4260110339188123`*^9, 3.4260110350404253`*^9}, {3.4260110821081047`*^9, 
  3.426011084611705*^9}, {3.4260111846155033`*^9, 3.426011213817494*^9}},
 FormatType->"TextForm"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"SetDirectory", "[", 
      RowBox[{"ToFileName", "[", 
       RowBox[{"Extract", "[", 
        RowBox[{
         RowBox[{"\"\<FileName\>\"", "/.", "\[InvisibleSpace]", 
          RowBox[{"NotebookInformation", "[", 
           RowBox[{"EvaluationNotebook", "[", "]"}], "]"}]}], ",", 
         RowBox[{"{", "1", "}"}], ",", "FrontEnd`FileName"}], "]"}], "]"}], 
      "]"}], ";"}], ")"}], " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   
   RowBox[{"(*", " ", 
    RowBox[{"Define", " ", "a", " ", "zero", " ", "tolerance"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"eps", "=", "5.`*^-6"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"B", "-", 
     RowBox[{"spline", " ", "evaluation"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"spanleft", "[", 
      RowBox[{"n_", ",", "knots_", ",", "i_", ",", "x_"}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Abs", "[", 
         RowBox[{
          RowBox[{"knots", "\[LeftDoubleBracket]", 
           RowBox[{"i", "+", "n", "+", "1"}], "\[RightDoubleBracket]"}], "-", 
          
          RowBox[{"knots", "\[LeftDoubleBracket]", 
           RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}]}], "]"}], "<", 
        "eps"}], ",", "0", ",", 
       FractionBox[
        RowBox[{"x", "-", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}]}], 
        RowBox[{
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "n", "+", "1"}], "\[RightDoubleBracket]"}], "-", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}]}]]}], "]"}]}], 
    ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"spanright", "[", 
     RowBox[{"n_", ",", "knots_", ",", "i_", ",", "x_"}], "]"}], ":=", 
    RowBox[{"If", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Abs", "[", 
        RowBox[{
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "n", "+", "2"}], "\[RightDoubleBracket]"}], "-", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "2"}], "\[RightDoubleBracket]"}]}], "]"}], "<", 
       "eps"}], ",", "0", ",", 
      FractionBox[
       RowBox[{
        RowBox[{"knots", "\[LeftDoubleBracket]", 
         RowBox[{"i", "+", "n", "+", "2"}], "\[RightDoubleBracket]"}], "-", 
        "x"}], 
       RowBox[{
        RowBox[{"knots", "\[LeftDoubleBracket]", 
         RowBox[{"i", "+", "n", "+", "2"}], "\[RightDoubleBracket]"}], "-", 
        RowBox[{"knots", "\[LeftDoubleBracket]", 
         RowBox[{"i", "+", "2"}], "\[RightDoubleBracket]"}]}]]}], "]"}]}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"bspline", "[", 
      RowBox[{"n_", ",", "knots_", ",", "i_", ",", "x_"}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "\[GreaterEqual]", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}]}], "&&", 
        RowBox[{"x", "<", 
         RowBox[{"knots", "\[LeftDoubleBracket]", 
          RowBox[{"i", "+", "2"}], "\[RightDoubleBracket]"}]}]}], ",", "1", 
       ",", "0"}], "]"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"bspline", "[", 
      RowBox[{"n_", ",", "knots_", ",", "i_", ",", "x_"}], "]"}], ":=", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"spanleft", "[", 
         RowBox[{"n", ",", "knots", ",", "i", ",", "x"}], "]"}], " ", 
        RowBox[{"bspline", "[", 
         RowBox[{
          RowBox[{"n", "-", "1"}], ",", "knots", ",", "i", ",", "x"}], 
         "]"}]}], "+", 
       RowBox[{
        RowBox[{"spanright", "[", 
         RowBox[{"n", ",", "knots", ",", "i", ",", "x"}], "]"}], " ", 
        RowBox[{"bspline", "[", 
         RowBox[{
          RowBox[{"n", "-", "1"}], ",", "knots", ",", 
          RowBox[{"i", "+", "1"}], ",", "x"}], "]"}]}]}], "/;", 
      RowBox[{"n", ">", "0"}]}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Degree", " ", "3", " ", "B"}], "-", "spline"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"n", "=", "3"}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Knot", " ", "sequence"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Play", " ", "with", " ", "adding", " ", "a", " ", "knot", " ", "at", " ",
      
     RowBox[{"13.5", " ", "--"}], " ", "it", " ", "will", " ", "improve", " ",
      "the", " ", "results"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"u", "=", 
     RowBox[{"{", 
      RowBox[{
      "0", ",", "0", ",", "0", ",", "0", ",", "4", ",", "8", ",", "12", ",", 
       "16", ",", "20", ",", "20", ",", "20", ",", "29"}], "}"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Places", " ", "to", " ", "create", " ", "data"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"v", "=", 
     RowBox[{"{", 
      RowBox[{
      "0", ",", "2", ",", "4", ",", "8", ",", "12", ",", "13.5", ",", " ", 
       "16", ",", "18", ",", "20"}], "}"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"ldata", "=", 
     RowBox[{"Length", "[", "v", "]"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"k", "=", 
     RowBox[{
      RowBox[{"Length", "[", "u", "]"}], "-", "3"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Number", " ", "of", " ", "control", " ", 
     RowBox[{"points", " ", "/", " ", "coefficients"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"l", "=", 
     RowBox[{"k", "-", "n", "+", "1"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"B", "-", 
     RowBox[{"spline", " ", "curve"}]}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"bsfunct", "[", 
      RowBox[{"n_", ",", "u_", ",", "d_", ",", "l_", ",", "x_"}], "]"}], ":=",
      
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "0"}], "l"], 
      RowBox[{
       RowBox[{"d", "\[LeftDoubleBracket]", 
        RowBox[{"i", "+", "1"}], "\[RightDoubleBracket]"}], " ", 
       RowBox[{"bspline", "[", 
        RowBox[{"n", ",", "u", ",", "i", ",", "x"}], "]"}]}]}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Set", " ", "up", " ", "design", " ", "matrix"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Evaluation", " ", "of", " ", "B"}], "-", 
     RowBox[{"splines", " ", "at", " ", "each", " ", 
      RowBox[{"data", "'"}], "s", " ", "knot", " ", "value"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Vander", "[", 
      RowBox[{"n_", ",", "l_", ",", "u_", ",", "ldata_", ",", "v_"}], "]"}], ":=",
      
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"bspline", "[", 
        RowBox[{"3", ",", "u", ",", "i", ",", 
         RowBox[{
         "v", "\[LeftDoubleBracket]", "j", "\[RightDoubleBracket]"}]}], "]"}],
        ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "1", ",", "ldata"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "0", ",", "l"}], "}"}]}], "]"}]}], ")"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Function", " ", "from", " ", "which", " ", "we", " ", "will", " ", 
     "sample"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"f", "[", "x_", "]"}], ":=", 
      RowBox[{"N", "[", 
       RowBox[{
        SqrtBox["x"], "+", 
        FractionBox[
         RowBox[{"Sin", "[", "x", "]"}], "10"]}], "]"}]}], ";"}], ")"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Form", " ", "matrix", " ", "part", " ", "of", " ", "normal", " ", 
     "equations"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"Mat", "[", 
       RowBox[{"n_", ",", "l_", ",", "u_", ",", "ldata_", ",", "v_"}], "]"}], 
      ":=", 
      RowBox[{
       RowBox[{"Transpose", "[", 
        RowBox[{"Vander", "[", 
         RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], 
        "]"}], ".", 
       RowBox[{"Vander", "[", 
        RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}]}]}], 
     ";"}], ")"}], " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Evauate", " ", "function", " ", "at", " ", "associated", " ", "knot", 
     " ", "values"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"data", "[", "ldata_", "]"}], ":=", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"f", "[", 
         RowBox[{"v", "\[LeftDoubleBracket]", "j", "\[RightDoubleBracket]"}], 
         "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"j", ",", "1", ",", "ldata"}], "}"}]}], "]"}]}], ";"}], 
    ")"}], " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Solve", " ", "for", " ", "coefficients", " ", "of", " ", "B"}], 
     "-", 
     RowBox[{"spline", " ", "curve"}]}], "  ", "*)"}], 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"bvals", "[", 
       RowBox[{"n_", ",", "l_", ",", "u_", ",", "ldata_", ",", "v_"}], "]"}], 
      ":=", 
      RowBox[{
       RowBox[{"Inverse", "[", 
        RowBox[{"Mat", "[", 
         RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], 
        "]"}], ".", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"Transpose", "[", 
          RowBox[{"Vander", "[", 
           RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], 
          "]"}], ".", 
         RowBox[{"data", "[", "ldata", "]"}]}], ")"}]}]}], ";"}], ")"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"d", " ", "stores", " ", "the", " ", "B"}], "-", 
     RowBox[{"spline", " ", "coefficients"}]}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"d", "=", 
      RowBox[{"N", "[", 
       RowBox[{"bvals", "[", 
        RowBox[{"n", ",", "l", ",", "u", ",", "ldata", ",", "v"}], "]"}], 
       "]"}]}], ";"}], ")"}], " ", "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Print", "[", 
      RowBox[{"\"\<B-spline coeffs d: \>\"", ",", "d"}], "]"}], ";"}], ")"}], 
   " ", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Organize", " ", "the", " ", "knot", " ", "vector", " ", "locations", " ",
      "and", " ", "associated", " ", "function", " ", "values", " ", "in", 
     " ", "a", " ", "table"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"vals", "=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"v", "\[LeftDoubleBracket]", "i", "\[RightDoubleBracket]"}], 
         ",", 
         RowBox[{"f", "[", 
          RowBox[{"v", "\[LeftDoubleBracket]", "i", "\[RightDoubleBracket]"}],
           "]"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "1", ",", "ldata"}], "}"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Print", "[", 
     RowBox[{
     "\"\<ldata = \>\"", ",", "ldata", ",", "\"\<  vals = \>\"", ",", 
      "vals"}], "]"}], ";"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Plot", " ", "the", " ", "points", " ", "in", " ", "vals"}], " ", 
    "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"ListPlot", "[", 
    RowBox[{"vals", ",", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"PointSize", "[", "0.03`", "]"}]}], ",", 
     RowBox[{"BaseStyle", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"\"\<FontFamily\>\"", "\[Rule]", "\"\<Helvetica\>\""}], ",", 
        RowBox[{"\"\<FontSize\>\"", "\[Rule]", "11"}]}], "}"}]}]}], "]"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"PLT", "=", 
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"f", "[", "x", "]"}], ",", 
        RowBox[{"bsfunct", "[", 
         RowBox[{"n", ",", "u", ",", "d", ",", "l", ",", "x"}], "]"}]}], 
       "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"v", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
        ",", 
        RowBox[{
        "v", "\[LeftDoubleBracket]", "ldata", "\[RightDoubleBracket]"}]}], 
       "}"}], ",", 
      RowBox[{"PlotStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Thickness", "[", "0.01`", "]"}], ",", 
           RowBox[{"GrayLevel", "[", "0.5`", "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"Thickness", "[", "0.01`", "]"}], "}"}]}], "}"}]}], ",", 
      RowBox[{"Axes", "\[Rule]", "False"}], ",", 
      RowBox[{"BaseStyle", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"\"\<FontFamily\>\"", "\[Rule]", "\"\<Helvetica\>\""}], ",", 
         
         RowBox[{"\"\<FontSize\>\"", "\[Rule]", "11"}]}], "}"}]}]}], "]"}]}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"showit", "=", 
    RowBox[{"Show", "[", 
     RowBox[{"{", 
      RowBox[{"points", ",", "PLT"}], "}"}], "]"}]}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Output", " ", "to", " ", "a", " ", "postscript", " ", "file"}], 
    " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Export", "[", 
      RowBox[{"\"\<splint1a.eps\>\"", ",", "showit"}], "]"}], ";"}], " ", 
    "*)"}]}]}]], "Input",
 CellChangeTimes->{{3.4260073582835083`*^9, 3.4260075439104266`*^9}, {
   3.4260093615941277`*^9, 3.426009513823022*^9}, {3.426009548693163*^9, 
   3.42600979017039*^9}, 3.426009849065076*^9, {3.426009881161228*^9, 
   3.4260099347182393`*^9}, {3.4260099828374314`*^9, 3.426010012339854*^9}, {
   3.4260103741701403`*^9, 3.426010422800067*^9}, {3.426010503395958*^9, 
   3.426010503806548*^9}, {3.4260106085371437`*^9, 3.4260106287061453`*^9}, {
   3.426010703914289*^9, 3.4260108348425546`*^9}, {3.4260109174613543`*^9, 
   3.426010943288492*^9}, {3.426013211169545*^9, 3.4260132301568475`*^9}, {
   3.426013274290308*^9, 3.4260133021103115`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"B-spline coeffs d: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{
    "0.00017422377148649073`", ",", "1.679406114737451`", ",", 
     "1.6965762512351337`", ",", "3.1086539613355604`", ",", 
     "3.3765370979121694`", ",", "4.159375690459903`", ",", 
     "4.089308003805246`", ",", "4.565346941558693`"}], "}"}]}],
  SequenceForm[
  "B-spline coeffs d: ", {0.00017422377148649073`, 1.679406114737451, 
   1.6965762512351337`, 3.1086539613355604`, 3.3765370979121694`, 
   4.159375690459903, 4.089308003805246, 4.565346941558693}],
  Editable->False]], "Print",
 CellChangeTimes->{
  3.4260107608862104`*^9, {3.426010799071118*^9, 3.4260108357438507`*^9}, {
   3.4260109196344795`*^9, 3.4260109443700476`*^9}, 3.4260132318292522`*^9, 
   3.426013281590806*^9}],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"ldata = \"\>", "\[InvisibleSpace]", "9", 
   "\[InvisibleSpace]", "\<\"  vals = \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "0.`"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "1.5051433050556633`"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "1.9243197504692071`"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "2.9273629494085283`"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"12", ",", "3.410444323337711`"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"13.5`", ",", "3.754613056829929`"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"16", ",", "3.9712096683334934`"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"18", ",", "4.167541962442118`"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"20", ",", "4.563430480072342`"}], "}"}]}], "}"}]}],
  SequenceForm[
  "ldata = ", 9, 
   "  vals = ", {{0, 0.}, {2, 1.5051433050556633`}, {
    4, 1.9243197504692071`}, {8, 2.9273629494085283`}, {
    12, 3.410444323337711}, {13.5, 3.754613056829929}, {
    16, 3.9712096683334934`}, {18, 4.167541962442118}, {
    20, 4.563430480072342}}],
  Editable->False]], "Print",
 CellChangeTimes->{
  3.4260107608862104`*^9, {3.426010799071118*^9, 3.4260108357438507`*^9}, {
   3.4260109196344795`*^9, 3.4260109443700476`*^9}, 3.4260132318292522`*^9, 
   3.4260132816108346`*^9}]
}, Open  ]],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointSize[0.03], 
    PointBox[{{0., 0.}, {2., 1.5051433050556633`}, {4., 
     1.9243197504692071`}, {8., 2.9273629494085283`}, {12., 
     3.410444323337711}, {13.5, 3.754613056829929}, {16., 
     3.9712096683334934`}, {18., 4.167541962442118}, {20., 
     4.563430480072342}}]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  BaseStyle->{"FontFamily" -> "Helvetica", "FontSize" -> 11},
  PlotRange->{{0., 20.}, {0., 4.563430480072342}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{{3.4260109196845512`*^9, 3.426010944430134*^9}, 
   3.426013231879324*^9, 3.4260132816508923`*^9}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {GrayLevel[0.5], Thickness[0.01], LineBox[CompressedData["
1:eJwV1nc8Vf8fB/Br3YHuuSQqGTfXta97kbbPu4iSFBJlZVNGhEqaVEIRssoK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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.01], 
    LineBox[CompressedData["
1:eJwV03k01O/3APBhhCzNe9BiyVL2ZcZ7plCW50ollEqJJFshijZb0YciWoh8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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  AxesOrigin->{0, 0},
  BaseStyle->{"FontFamily" -> "Helvetica", "FontSize" -> 11},
  PlotRange->{{0, 20}, {0., 4.565346795832503}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{{3.4260109196845512`*^9, 3.426010944430134*^9}, 
   3.426013231879324*^9, 3.426013283403412*^9}],

Cell[BoxData[
 GraphicsBox[{{{}, 
    {Hue[0.67, 0.6, 0.6], PointSize[0.03], 
     PointBox[{{0., 0.}, {2., 1.5051433050556633`}, {4., 
      1.9243197504692071`}, {8., 2.9273629494085283`}, {12., 
      3.410444323337711}, {13.5, 3.754613056829929}, {16., 
      3.9712096683334934`}, {18., 4.167541962442118}, {20., 
      4.563430480072342}}]}, {}}, {{}, {}, 
    {GrayLevel[0.5], Thickness[0.01], LineBox[CompressedData["
1:eJwV1nc8Vf8fB/Br3YHuuSQqGTfXta97kbbPu4iSFBJlZVNGhEqaVEIRssoK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      "]]}, 
    {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.01], 
     LineBox[CompressedData["
1:eJwV03k01O/3APBhhCzNe9BiyVL2ZcZ7plCW50ollEqJJFshijZb0YciWoh8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      "]]}}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  BaseStyle->{"FontFamily" -> "Helvetica", "FontSize" -> 11},
  PlotRange->{{0., 20.}, {0., 4.563430480072342}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{{3.4260109196845512`*^9, 3.426010944430134*^9}, 
   3.426013231879324*^9, 3.4260132834334555`*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
},
WindowSize->{830, 891},
WindowMargins->{{10, Automatic}, {Automatic, -11}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 1029, 28, 454, "Input"],
Cell[CellGroupData[{
Cell[1622, 53, 15110, 386, 2152, "Input"],
Cell[CellGroupData[{
Cell[16757, 443, 823, 17, 50, "Print"],
Cell[17583, 462, 1429, 35, 50, "Print"]
}, Open  ]],
Cell[19027, 500, 772, 18, 322, "Output"],
Cell[19802, 520, 12835, 218, 309, "Output"],
Cell[32640, 740, 23325, 393, 322, 13067, 223, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
