(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     37778,        747]
NotebookOptionsPosition[     37248,        727]
NotebookOutlinePosition[     37667,        743]
CellTagsIndexPosition[     37624,        740]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell["\<\

Mathematical Principles for Scientific Computing and Visualization  
http://www.farinhansford.com/books/scv
by Gerald Farin and Dianne Hansford

This notebook demonstrates stability problems with the monomial (1,t, t^2, \
t^3,....) polynomials.

The Bernstein polynomials, defined below, are a more stable set of basis \
functions. Below you will find a plot of a degree 14 polynomials -- in \
monomial form and in Bernstein form.
A polynomial in terms of the Bernstein basis is a Bezier function.

For more on Bezier curves, see Farin and Hansford's Essentials of CAGD.


See Chapter 2: Computational Basics for details.

Updated May 2008\
\>", "Input",
 CellChangeTimes->{{3.420482125260645*^9, 3.4204822490987153`*^9}, {
  3.4204823768624306`*^9, 3.420482407216077*^9}, {3.420482440423827*^9, 
  3.420482462175104*^9}},
 FormatType->"TextForm"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{"(*", " ", 
   RowBox[{
   "Define", " ", "the", " ", "Bernstein", " ", "polynomials", " ", 
    "recursively"}], " ", "*)"}], "\[IndentingNewLine]", 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"Bern", "[", 
     RowBox[{"n_", ",", "0", ",", "t_"}], "]"}], ":=", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", "t"}], ")"}], "^", "n"}]}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Bern", "[", 
     RowBox[{"n_", ",", "n_", ",", "t_"}], "]"}], ":=", 
    RowBox[{"t", "^", "n"}]}], "\[IndentingNewLine]", 
   StyleBox[
    RowBox[{
     RowBox[{"Bern", "[", 
      RowBox[{"n_", ",", "i_", ",", "t_"}], "]"}], " ", ":=", " ", 
     RowBox[{
      RowBox[{"Binomial", "[", 
       RowBox[{"n", ",", "i"}], "]"}], " ", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"1", "-", "t"}], ")"}], "^", 
       RowBox[{"(", 
        RowBox[{"n", "-", "i"}], ")"}]}], "  ", 
      RowBox[{"t", "^", "i"}]}]}],
    FontFamily->"Arial"], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Create", " ", "a", " ", "table", " ", "of", " ", "degree", " ", "n", " ",
      "Bernstein", " ", "polynomials", " ", "for", " ", "paramter", " ", 
     "t"}], " ", "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   StyleBox[
    RowBox[{
     RowBox[{"Bernvec", "[", 
      RowBox[{"n_", ",", "t_"}], "]"}], ":=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Bern", "[", 
        RowBox[{"n", ",", "i", ",", "t"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "0", ",", "n"}], "}"}]}], "]"}]}],
    FontFamily->"Arial"], 
   StyleBox["\[IndentingNewLine]",
    FontFamily->"Arial"], "\[IndentingNewLine]", 
   StyleBox[
    RowBox[{
     StyleBox[
      RowBox[{"(", "*"}]], " ", 
     RowBox[{
     "Evaluate", " ", "the", " ", "degree", " ", "n", " ", "Bezier", " ", 
      "function", " ", "at", " ", "t"}], " ", 
     StyleBox[
      RowBox[{"*", ")"}]]}],
    FontFamily->"Arial"], "\[IndentingNewLine]", 
   StyleBox[
    RowBox[{
     RowBox[{
      RowBox[{"Bez", "[", 
       RowBox[{"b_", ",", "n_", ",", "t_"}], "]"}], ":=", 
      RowBox[{
       RowBox[{"Bernvec", "[", 
        RowBox[{"n", ",", "t"}], "]"}], ".", "b"}]}], ";"}],
    FontFamily->"Arial"], "\[IndentingNewLine]", 
   StyleBox["\[IndentingNewLine]",
    FontFamily->"Arial"], 
   StyleBox[
    RowBox[{
     RowBox[{"b", "=", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1"}], ",", "1", ",", 
        RowBox[{"-", "2"}], ",", "3", ",", 
        RowBox[{"-", "4"}], ",", "5", ",", 
        RowBox[{"-", "6"}], ",", "7", ",", 
        RowBox[{"-", "6"}], ",", "5", ",", 
        RowBox[{"-", "4"}], ",", "3", ",", 
        RowBox[{"-", "2"}], ",", "1", ",", "0"}], "}"}]}], ";"}],
    FontFamily->"Arial"], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Define", " ", "a", " ", "degree", " ", "14", " ", "polynomial", " ", 
     "and", " ", "print", " ", "it", " ", "expanded", " ", "in", " ", 
     "monomial", " ", "form"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "Mathematica", " ", "combines", " ", "terms", " ", "of", " ", "common", 
     " ", 
     RowBox[{"t", "^", "k"}]}], " ", "*)"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"poly", "[", "t_", "]"}], "=", 
    StyleBox[
     RowBox[{"Expand", "[", 
      RowBox[{"Bez", "[", 
       RowBox[{"b", ",", "14", ",", "t"}], "]"}], "]"}],
     FontFamily->"Arial"]}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "The", " ", "degree", " ", "14", " ", "monomial", " ", "polynomial", " ", 
     "is", " ", "defined", " ", "as", " ", "such", " ", "here"}], " ", "*)"}],
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"poly1", "[", "t_", "]"}], ":=", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "90665"}], " ", 
        SuperscriptBox["t", "14"]}], "+", 
       RowBox[{"634662", " ", 
        SuperscriptBox["t", "13"]}], "-", 
       RowBox[{"2020655", " ", 
        SuperscriptBox["t", "12"]}], "+", 
       RowBox[{"3873324", " ", 
        SuperscriptBox["t", "11"]}], "-", 
       RowBox[{"4989985", " ", 
        SuperscriptBox["t", "10"]}], "+", 
       RowBox[{"4.5705661`*^6", " ", 
        SuperscriptBox["t", "9"]}], "-", 
       RowBox[{"3072069", " ", 
        SuperscriptBox["t", "8"]}], "+", 
       RowBox[{"1540968", " ", 
        SuperscriptBox["t", "7"]}], "-", 
       RowBox[{"579579", " ", 
        SuperscriptBox["t", "6"]}], "+", 
       RowBox[{"162162", " ", 
        SuperscriptBox["t", "5"]}], "-", 
       RowBox[{"33033", " ", 
        SuperscriptBox["t", "4"]}], "+", 
       RowBox[{"4732", " ", 
        SuperscriptBox["t", "3"]}], "-", 
       RowBox[{"455", " ", 
        SuperscriptBox["t", "2"]}], "+", 
       RowBox[{"28", " ", "t"}], "-", "1"}]}], ";"}], ")"}], " ", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Plot", " ", "the", " ", "two", " ", "functions"}], " ", "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "The", " ", "monomial", " ", "form", " ", "becomes", " ", "numerically", 
     " ", "unstable", " ", "away", " ", "from", " ", "0"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"poly", "[", "t", "]"}], ",", 
       RowBox[{"poly1", "[", "t", "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "1.1"}], "}"}]}], "]"}], " "}]}]], "Input",
 CellChangeTimes->{{3.4204822531345186`*^9, 3.420482265722619*^9}, {
  3.420482314713064*^9, 3.4204823667579007`*^9}, {3.4204825038350077`*^9, 
  3.420482683012653*^9}, {3.420482722689706*^9, 3.4204827967361794`*^9}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1"}], "+", 
  RowBox[{"28", " ", "t"}], "-", 
  RowBox[{"455", " ", 
   SuperscriptBox["t", "2"]}], "+", 
  RowBox[{"4732", " ", 
   SuperscriptBox["t", "3"]}], "-", 
  RowBox[{"33033", " ", 
   SuperscriptBox["t", "4"]}], "+", 
  RowBox[{"162162", " ", 
   SuperscriptBox["t", "5"]}], "-", 
  RowBox[{"579579", " ", 
   SuperscriptBox["t", "6"]}], "+", 
  RowBox[{"1540968", " ", 
   SuperscriptBox["t", "7"]}], "-", 
  RowBox[{"3072069", " ", 
   SuperscriptBox["t", "8"]}], "+", 
  RowBox[{"4570566", " ", 
   SuperscriptBox["t", "9"]}], "-", 
  RowBox[{"4989985", " ", 
   SuperscriptBox["t", "10"]}], "+", 
  RowBox[{"3873324", " ", 
   SuperscriptBox["t", "11"]}], "-", 
  RowBox[{"2020655", " ", 
   SuperscriptBox["t", "12"]}], "+", 
  RowBox[{"634662", " ", 
   SuperscriptBox["t", "13"]}], "-", 
  RowBox[{"90665", " ", 
   SuperscriptBox["t", "14"]}]}]], "Output",
 CellChangeTimes->{{3.4204827063462048`*^9, 3.420482746053301*^9}, {
  3.420482789575883*^9, 3.420482797527317*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJw92nk0VV8bB/CboYSQkLlIKUQhIfVIZExUxhAZIpUS0qUMEZmnTClzhkQk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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], LineBox[CompressedData["
1:eJw92nk0Vd/7B/AbCpEklMjQQEkImRoeyTw1KlQi0aCQVIokqXxI3EtEZMic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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0, 1.1}, {-0.5220813199780214, 0.26423774869181216`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{{3.4204827063462048`*^9, 3.420482746053301*^9}, {
  3.420482789575883*^9, 3.4204827976174464`*^9}},
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
},
WindowSize->{683, 754},
WindowMargins->{{Automatic, 32}, {Automatic, 15}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 858, 24, 296, "Input"],
Cell[CellGroupData[{
Cell[1451, 49, 5930, 159, 648, "Input"],
Cell[7384, 210, 1032, 31, 60, "Output"],
Cell[8419, 243, 28813, 481, 238, 21755, 364, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
